/****************************************
 * 
 * Emelmovebar - a plugin for emelFM
 * by Florian Zaehringer
 *
 * source of source-code and inspiration: 
 *
 * emelFM by Michael Clark
 *
 * *************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ***************************************/


#include "../emelfm.h"

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <pthread.h>


#define MAX_CHAR 55
#define BARRES 10000

struct _progbarstruct
{
	gint process_id; 		/* process's PID (0 = cancel) */
	gint file_sum;				/*total amount of files (0 = cancel) */
	gchar *currname;

}progbarstruct;

/* functions taken from emelfm */

/* du.c and ripped appart*/
guint64
disk_usage(gchar *filename)
{
  struct stat statbuf;
	guint64 total=0;

  if (lstat(filename, &statbuf) != 0)
  {
    status_errno();
    return total;
  }
  total += statbuf.st_size;
  if (S_ISDIR(statbuf.st_mode))
  {
    DIR *dp;
    struct dirent *entry;
    struct stat statbuf;
    gchar path[PATH_MAX];

    if ((dp = opendir(filename)) == NULL)
    {
      status_errno();
      status_message("Warning: couldn't open directory ");
      status_message(filename);
      status_message("\n");
      return total;
    }

    while ((entry = readdir(dp)) != NULL)
    {
      if (STREQ(entry->d_name, ".") || STREQ(entry->d_name, ".."))
        continue;

      g_snprintf(path, PATH_MAX, "%s/%s", filename, entry->d_name);
      if (lstat(path, &statbuf) != 0)
      {
        status_errno();
        continue;
      }

      total += statbuf.st_size;
      if (S_ISDIR(statbuf.st_mode))
        total += disk_usage(path);
    }
    closedir(dp);
  }
	return total;
}

/* 
 * had to modify the next two functions;
 * the original ones would produce nasty 
 * --end-of-output--
 */
static void
start_reader_thread_progbar(void *ptr)
{
  gchar buf[1024];
  gint nread;
  gint *file_pipe = (gint *)ptr;

  while ((nread = read(*file_pipe, buf, sizeof(buf)-1)) > 0)
  {
    buf[nread] = '\0';
    gdk_threads_enter();
    gtk_text_insert(GTK_TEXT(app.output_text), app.output_font, NULL, NULL,
                    buf, -1);
    gdk_threads_leave();
  }
  close(*file_pipe);
  g_free(file_pipe);
  pthread_exit(0);
}

gint
exec_and_capture_output_threaded_progbar(gchar *command)
{
  gint file_pipes[2];
  gint pid;
  gchar *args[4];
  pthread_t reader_thread;


  if (pipe(file_pipes) != 0)
  {
    status_errno();
    return -1;
  }

  if ((pid = fork()) == -1)
  {
    status_errno();
    return -1;
  }

  if (pid == 0)
  {
    close(0);
    close(1);
    close(2);
    dup(file_pipes[0]);
    dup(file_pipes[1]);
    dup(file_pipes[1]);
    close(file_pipes[0]);
    close(file_pipes[1]);

    args[0] = "sh";
    args[1] = "-c";
    args[2] = command;
    args[3] = NULL;
    execvp(args[0], args);
    exit(127);
  }
  else
  {
    close(file_pipes[1]);
    pthread_create(&reader_thread, NULL, (void *)&start_reader_thread_progbar,
                   g_memdup(&file_pipes[0], sizeof(gint)));
  }

  return pid;
}

/* end of emelfm functions */


static void
copy_break()
{
	gchar command[MAX_LEN];
	
	if (progbarstruct.process_id != 0)
	{
		kill(progbarstruct.process_id ,SIGKILL);
		g_snprintf(command, sizeof(command), "rm -rf \"%s\"", progbarstruct.currname);
		exec_and_capture_output_threaded_progbar(command);
		progbarstruct.process_id = 0;
	}
	return;
}

static void
break_all()
{
	progbarstruct.file_sum = 0;
	copy_break(); 
}


static void 
shorten(gchar *string, gchar *dots, guint *string_diff)
{
	if (strlen(string) > MAX_CHAR)
	{
		*string_diff	= strlen(string) - MAX_CHAR;
		strcpy(dots, "...");
	}
	else
	{
		*string_diff = 0;
		strcpy(dots, "");
	}
	return;
}

static void 
mvbar()
{
  gchar src[PATH_MAX+NAME_MAX], dest[PATH_MAX+NAME_MAX];
	gchar command[MAX_LEN];
	gchar src_dots[4], dest_dots[4];
	gchar labeltext[220]="Initializing emelmovebar...";
	gchar progbartext[64]="";
  gchar *s;
	gchar *src_dir, *dest_dir;

	gint i = 0;
	gint status;
  gint overwrite_button;
	guint src_diff, dest_diff;
	guint64 currentsize = 0;
	guint64 totalsrcssize = 1; /*prevent /0 */ 
	guint64 totaltargetsize = 0;
	guint64 percent = 0;
  gboolean check = cfg.confirm_overwrite;
  
  GList *snapshot_list = NULL;
  GList *const_list = NULL;
	GtkWidget *window;
	GtkWidget *button;
	GtkWidget *separator;
	GtkWidget *vbox, *hbox;
	GtkWidget *label;
	GtkWidget *progbar;
	GtkAdjustment *adj;
	

	if ((access(other_view->dir, W_OK) == -1))
	{
		status_message("\n"
				"Sorry, I recently spoke to your filesystem and it told me\n"
				"you wouldn't be allowed to write there. You should check that...\n");
		return;
	}

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), "emelmovebar");
	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
	gtk_widget_set_usize(GTK_WIDGET(window), 500, 150);
	gtk_signal_connect(GTK_OBJECT(window), "delete_event", GTK_SIGNAL_FUNC(break_all), NULL);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);
	gtk_container_add(GTK_CONTAINER(window), vbox);
	gtk_widget_show(vbox);

	adj = (GtkAdjustment *) gtk_adjustment_new(0, 1, BARRES, 0, 0, 0); 

	progbar = gtk_progress_bar_new_with_adjustment (adj);
	gtk_box_pack_start(GTK_BOX(vbox), progbar, TRUE, TRUE, 5);
	gtk_progress_set_show_text(GTK_PROGRESS(progbar), TRUE);
	gtk_progress_set_format_string(GTK_PROGRESS(progbar), progbartext);
	gtk_widget_show(progbar);

	label = gtk_label_new(labeltext);
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
	gtk_widget_show(label);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, TRUE, TRUE, 0);
	gtk_widget_show(separator);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show(hbox);

	button = gtk_button_new_with_label("   abort   ");
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(break_all),
			NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 0);
	gtk_widget_show(button);

	button = gtk_button_new_with_label("   skip  ");
	gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(copy_break),
			NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 0);
	gtk_widget_show(button);


	progbarstruct.process_id = 0;
	progbarstruct.file_sum = g_list_length(get_selection(curr_view));
	snapshot_list = get_selection(curr_view);
	src_dir = g_strdup(curr_view->dir);
	dest_dir = g_strdup(other_view->dir);

	/*maybe pull totalsrcsize out of there... to be quicker if user changes dirs*/
	for (; (snapshot_list != NULL) ; snapshot_list = snapshot_list->next)
	{
		totalsrcssize += disk_usage(((FileInfo *)snapshot_list->data)->filename);
		const_list = g_list_append(const_list, 
				(gpointer*) g_strdup(((FileInfo *)snapshot_list->data)->filename));
	}

	g_list_free(snapshot_list);
	if (progbarstruct.file_sum > 1) {
  	gtk_clist_unselect_all(GTK_CLIST(curr_view->clist));
	}
  refresh_list(other_view);
  refresh_list(other_view);

	for (; (const_list != NULL) && (progbarstruct.file_sum != 0); const_list = const_list->next)
	{
		i += 1;
		g_snprintf(dest, sizeof(dest), "%s/%s", dest_dir,
				((FileInfo *)const_list->data)->filename);
		g_snprintf(src, sizeof(src), "%s/%s", src_dir,
				((FileInfo *)const_list->data)->filename);
		g_snprintf(command, sizeof(command), "mv -f \"%s\" \"%s\"", src, dest);
		progbarstruct.currname = dest;

		
		if (check && (access(dest, F_OK) == 0))
		{
			if ((s = strrchr(dest, '/')) != NULL)
				create_confirm_overwrite_dialog(s+1, &overwrite_button);
			else
				create_confirm_overwrite_dialog(dest, &overwrite_button);

			gtk_widget_set_sensitive(app.main_window, FALSE);
			set_cursor(GDK_WATCH);
			gtk_main();
			set_cursor(GDK_LEFT_PTR);
			switch (overwrite_button)
			{
				case YES_TO_ALL:
					check = FALSE;
				case YES:
					progbarstruct.process_id = exec_and_capture_output_threaded_progbar(command);	
					gtk_widget_show(window);
				case NO:
				case CANCEL:
					break;
				default:
					fprintf(stderr, _("Something went wrong in plugin cobybar.\n"));
					break;
			}
			if (overwrite_button == CANCEL)
				break;
			else if (overwrite_button == NO)
				continue;
		}
    else
		{
			progbarstruct.process_id = exec_and_capture_output_threaded_progbar(command);	
			gtk_widget_show(window);
		}
				
		shorten((((FileInfo *)const_list->data)->filename), src_dots, &src_diff);
		shorten(dest_dir, dest_dots, &dest_diff);

		while (access(dest, F_OK) == -1)
			usleep(1000);
		
		while ((progbarstruct.process_id != 0))
		{
			currentsize = disk_usage(dest) + totaltargetsize;
			percent = (((guint64) (currentsize))*BARRES)/ ((guint64) totalsrcssize);

			g_snprintf(labeltext, sizeof (labeltext), 
					"move %s%s\nto %s%s "
					"\n"
					" moving file/directory %d of %d", 
					src_dots, ((((FileInfo *)const_list->data)->filename)+src_diff),
					dest_dots, (dest_dir+dest_diff), 
					i, progbarstruct.file_sum);
			g_snprintf(progbartext, sizeof (progbartext), 
					"%.2f MB of total %.2f MB   (%li %%\%%)",
					/* just for a %...  ugly, isnt it? any idea? */
					(currentsize / 1048576.0), (totalsrcssize / 1048576.0), 
					(((glong) percent)*100/BARRES));

			gtk_progress_set_value (GTK_PROGRESS( progbar ), percent);		
			gtk_label_set_text(GTK_LABEL(label), labeltext);
			gtk_progress_set_format_string(GTK_PROGRESS( progbar ), progbartext);

			while (gtk_events_pending())
				gtk_main_iteration();

			usleep(150000); /* cool down, baby */

			if (waitpid(progbarstruct.process_id, &status, WNOHANG)!=0)
			{
				totaltargetsize += disk_usage(dest); 
				/* why dest not src? to avoid errors if mv is too fast.*/
				break;
			}

			if (progbarstruct.process_id == 0)
			{
				totalsrcssize -= disk_usage(src);
			}
		}
	}
	g_list_free(const_list);
	g_free(src_dir);
	g_free(dest_dir);
	gtk_widget_destroy(window);
  refresh_list(other_view);
  refresh_list(other_view);
}

gint init_plugin(Plugin *p)
{
  p->name = "Moveprogressbar";
  p->description = "Move files with a progressbar";
  p->plugin_cb = mvbar;
                          
  return TRUE;
}
