/* 
 *  sgmixer.c
 *
 *  sGmixer - simple GTK2 audio mixer by mironcho
 *  (c) Miroslav Yordanov 2003  <miordanov@i-space.org>
 *				<mironcholinux@mail.bg>
 *  
 *  Thanks to Robert Homann for his patch.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.*
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>

#include "sgmixer.h"


gint mf, devmask, i;
static gchar filename[100];
gchar labels[ SOUND_MIXER_NRDEVICES ][30] = SOUND_DEVICE_LABELS;
gint values[ SOUND_MIXER_NRDEVICES ];			/* for current chanel levels */
gboolean muted[ SOUND_MIXER_NRDEVICES+1 ] = {FALSE};	/* current chanel status.
							the first value (muted[0]) 
							is reserved for main mutting
							with "mute" button from GUI */
GtkObject *adj[ SOUND_MIXER_NRDEVICES ];
static gboolean do_not_set_param=FALSE;




int main ( int argc, char **argv )
{
	GtkWidget *win;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *scale;
	GtkWidget *label;
	GtkWidget *name_button;
	GtkWidget *qbutton;
	GtkWidget *abutton;
	GtkWidget *mute_button;
	GtkWidget *sep;
	mixer_info info;
	int argindex;
	gboolean start_minimized=FALSE, do_load_from_file=TRUE;

	strncpy ( filename, getenv("HOME"), 89 );
	strncat ( filename, CONF_FILE, 10 );

	/* Mixer init */
	if ( -1 == (mf = open( MIXER, O_RDWR))) {
		g_warning ("Can't open %s !!! Quiting ...\n", MIXER);
		exit (1);
	}
	
	if ( -1 == (ioctl( mf, SOUND_MIXER_READ_DEVMASK, &devmask)) ) {
		g_warning ("SOUND_MIXER_READ_DEVMASK\n");
		exit (1);
	}
	
	if ( -1 == (ioctl( mf, SOUND_MIXER_INFO, &info )) ) {
		g_warning ("Can't read mixer info !!! Quiting ...\n");
		exit (1);
	}

	
	/* Gtk init */
	gtk_init ( &argc, &argv );

	for(argindex=1; argindex < argc; ++argindex)
	{
		if(strcmp(argv[argindex],"--minimized") == 0)
		{
			start_minimized=TRUE;
		}
		else if(strcmp(argv[argindex],"--read-from-mixer") == 0)
		{
			do_load_from_file=FALSE;
		}
		else
		{
			usage(argv[0]);
			exit(1);
		}
	}

	win = gtk_window_new ( GTK_WINDOW_TOPLEVEL );
	gtk_window_set_default_size (GTK_WINDOW (win), 250, 100);
	g_signal_connect ( G_OBJECT (win), "delete_event",
			G_CALLBACK (gtk_main_quit),
			NULL );
	
	vbox = gtk_vbox_new (TRUE, 0);
	gtk_container_add ( GTK_CONTAINER (win), vbox );
	gtk_container_set_border_width ( GTK_CONTAINER (win), 5 );

	
	/* Mixer name */
	label = gtk_label_new ( info.name );
	gtk_box_pack_start ( GTK_BOX (vbox), label, TRUE, TRUE, 1 );
	gtk_widget_show ( label );

	
	/* Creating scale widgets */
	for ( i = 0; i < SOUND_MIXER_NRDEVICES; i++) {
		if ( devmask & (1 << i) ) {
			adj[i] = gtk_adjustment_new ( 0.0, 0.0, 101.0, 1.0, 1.0, 1.0);
			set_adjustment_and_value(i);
	
			scale = gtk_hscale_new ( GTK_ADJUSTMENT (adj[i]) );
			GTK_SCALE (scale)->digits = 0;
			GTK_SCALE (scale)->value_pos = GTK_POS_RIGHT;

			g_signal_connect ( G_OBJECT (scale), "value_changed",
					G_CALLBACK (set_new_param),
					(gpointer) i );

			name_button = gtk_toggle_button_new_with_label ( labels[i] );
			gtk_button_set_relief ( GTK_BUTTON (name_button),
						GTK_RELIEF_NONE );
			gtk_widget_set_usize ( name_button, 65, 10);
			g_signal_connect ( G_OBJECT (name_button), "clicked",
					G_CALLBACK (mute_only),
					(gpointer) i );
			
			hbox = gtk_hbox_new (FALSE, 4);
			
			gtk_box_pack_start (GTK_BOX (hbox), name_button, FALSE, TRUE, 1);
			gtk_box_pack_start (GTK_BOX (hbox), scale, TRUE, TRUE, 1);
			gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 1);
			
			gtk_widget_show ( name_button );
			gtk_widget_show ( scale );
			gtk_widget_show ( hbox );

			/* initializing muted[i+1] */
			muted[i+1] = FALSE;
		}
	}

	/*  
	 *  After all adjustments and scales are created,
	 *  loading saved levels if conf file exists !
	 */
	if(do_load_from_file)
	{
		if ( !load_from_file() ) {
			g_message ("Error loading saved channels' values !!!\n");
		}
	}

	sep = gtk_hseparator_new ();
	gtk_box_pack_start (GTK_BOX (vbox), sep, TRUE, TRUE, 1);

	hbox = gtk_hbox_new (FALSE, 4);

	mute_button = gtk_toggle_button_new_with_label ("Mute all");
	g_signal_connect ( G_OBJECT (mute_button), "clicked",
			G_CALLBACK (mute_all),
			NULL );
	gtk_box_pack_start_defaults (GTK_BOX (hbox), mute_button );


	abutton = gtk_button_new_with_label ("About");
	g_signal_connect ( G_OBJECT (abutton), "clicked",
			G_CALLBACK (about_dialog),
			NULL );
	gtk_box_pack_start_defaults (GTK_BOX (hbox), abutton );


	qbutton = gtk_button_new_with_label ("Quit");
	g_signal_connect ( G_OBJECT (qbutton), "clicked",
			G_CALLBACK (gtk_main_quit),
			NULL );
	
	gtk_box_pack_start_defaults (GTK_BOX (hbox), qbutton );
	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 1);

	gtk_container_set_focus_child ( GTK_CONTAINER (win), qbutton );

	gtk_widget_show ( sep );
	gtk_widget_show ( mute_button );
	gtk_widget_show ( abutton );
	gtk_widget_show ( qbutton );
	gtk_widget_show ( hbox );
	gtk_widget_show ( vbox );
  
	if(start_minimized)
	{
		gtk_window_iconify(GTK_WINDOW(win));
	}
	
	gtk_widget_show ( win );
  
        g_timeout_add(POLL_INTERVALL,read_mixer_channels,NULL);

	gtk_main ();

	if ( !save_to_file() ) {
		g_message ("Error saving channels' levels !!!\n");
	}
	close ( mf );
	return 0;
}

/***********************************************************/
/*			functions			   */
/*		prototypes are in sgmixer.h		   */
/***********************************************************/


gint get_mixer_param( gint mdev )
{
	gint val = 0;
	
	if ( -1 == (ioctl( mf, MIXER_READ(mdev), &val)) ) {
		g_print ("Can't read from mixer !!! Quiting ...\n");
		exit (1);
	}

	val>>=8;

	// why ? ;)  because division by 256 was missing?
	if (val > 100) { 
		val = 100; 
	} else if (val < 0) {
		val = 0;
	}

	return val;
}


void set_mixer_param( gint mdev, gint val )
{
	int nval;
	nval = (val << 8) + val ;
	if ( -1 == (ioctl( mf, MIXER_WRITE(mdev), &nval )) ) {
		g_print ("Can't write to mixer !!! Quiting ...\n");
		exit (1);
	}
}


void set_new_param( GtkWidget *widget,
		gpointer data )
{
	if(do_not_set_param || muted[(gint) data])
	{
		return;
	}
	set_mixer_param ((gint)data, (gint) GTK_ADJUSTMENT (adj[(gint) data])->value );
	values[(gint) data] = GTK_ADJUSTMENT (adj[(gint) data])->value ;
}


void mute_all( GtkWidget *widget,
		gpointer data )
{
	gshort i;

	for ( i = 0; i < SOUND_MIXER_NRDEVICES; i++ ) {
		if (devmask & ( 1<<i ) ) {
			if ( GTK_TOGGLE_BUTTON (widget)->active ) {
				set_mixer_param( i, 0 );
				muted[0] = TRUE;
			} else {
				set_mixer_param( i, values[i] );
				muted[0] = FALSE;
			}
		}
	}
}


void mute_only( GtkWidget *widget,
		gpointer data )
{
	if ( GTK_TOGGLE_BUTTON (widget)->active ) {
		set_mixer_param( (gint) data, 0 );
		muted[(gint) data] = TRUE;
	} else {
		set_mixer_param( (gint) data, values[(gint) data] );
		muted[(gint) data] = FALSE;
	}
}


/*  
 *  save_to_file() and load_from_file() returns TRUE on success
 *  and FALSE on error !!
 */
gboolean save_to_file( void )
{
	FILE *fp;

	if ( NULL == (fp = fopen( filename, "wb"))) {
		return FALSE;
	}

	if ( 1 != fwrite ( values, sizeof(values), 1 , fp) ) {
		return FALSE;
	}

	fclose (fp);
	return TRUE;
}


gboolean load_from_file( void )
{
	FILE *fp;
	gshort i;

	if ( NULL == (fp = fopen( filename, "rb"))) {
		return FALSE;
	}

	if ( 1 != fread ( values, sizeof(values), 1, fp) ) {
		return FALSE;
	}

	for ( i = 0; i < SOUND_MIXER_NRDEVICES; i++ ) {
		if ( devmask & (1<<i) ) {
			set_mixer_param ( i, values[i] );
			GTK_ADJUSTMENT (adj[i])->value = values[i] ;
		}
	}

	return TRUE;
}


static void set_adjustment_and_value(gint i)
{
	values[i]=get_mixer_param(i);
	gtk_adjustment_set_value(GTK_ADJUSTMENT(adj[i]),values[i]);
}

static gboolean read_mixer_channels(gpointer data)
{
	do_not_set_param=TRUE;

	if ( muted[0] )
	{
		/* main mutting */
		do_not_set_param=FALSE;
		return TRUE;
	}
	
	for(i=0; i < SOUND_MIXER_NRDEVICES; ++i)
	{
		if ( muted[i+1] )
		{
			/* the current chanel is muted and we
			 * don't want to lose it's value 
			 * */
			do_not_set_param=FALSE;
			return TRUE;
		}
		
		if(devmask & (1 << i))
		{
			set_adjustment_and_value(i);
		}
	}
	do_not_set_param=FALSE;
	return TRUE;
}

static void usage(const char *progname)
{
	printf("Usage: %s [--minimized] [--read-from-mixer]\n"
		"  --minimized        start minimized\n"
		"  --read-from-mixer  read channel settings from mixer, not from config\n",
		progname);
}

/* EOF */

