/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "textundo.h"

#include <stdlib.h>

#include "glade_support.h"
#include "marshallers.h"

enum {
    PROP_BUFFER = 1,
    PROP_CAN_UNDO,
    PROP_CAN_REDO
};

static GQuark undo_data_quark;

#if 0
typedef struct {
    GList *list;
    gint length; /* Number of user actions not nodes (one action may be several nodes). */
} TextUndoDataList;
#endif /* 0 */

typedef GList *TextUndoDataList;

struct TextUndoData_
{
    GObject base;

    GtkTextBuffer *buffer;
    TextUndoDataList undo, redo;
    TextUndoDataList *p_current, *p_other;
    /* while undo operations p_current = &redo, all other time p_current = &undo. p_other is vice versa. */
    gint unmodified_point; /* >0 - in redo (future), <0 - in undo (past). */
    gint max_undo_steps; /* FIXME */
    gboolean now_user_action :1, now_undo_or_redo :1, user_action_non_empty :1;
};

struct TextUndoDataClass_
{
    GObjectClass base_class;

    void (*undo) (TextUndoData *undo_data);
    void (*redo) (TextUndoData *redo_data);
    void (*moved)(TextUndoData *undo_data);
};


static void text_undo_data_class_init(TextUndoDataClass *klass);
static void text_undo_data_init(TextUndoData *undo_data);
void text_undo_data_finalize(GObject *object);

static void text_undo_data_set_property(GObject      *object,
                                        guint         prop_id,
                                        const GValue *value,
                                        GParamSpec   *pspec);
static void text_undo_data_get_property(GObject      *object,
                                        guint         prop_id,
                                        GValue       *value,
                                        GParamSpec   *pspec);

GtkType text_undo_data_get_type(void)
{
    static GType tu_type = 0;

    if(!tu_type) {
        static const GTypeInfo tu_info = {
            sizeof(TextUndoDataClass),
            NULL,               /* base_init */
            NULL,               /* base_finalize */
            (GClassInitFunc)text_undo_data_class_init,
            NULL,               /* class_finalize */
            NULL,               /* class_data */
            sizeof(TextUndoData),
            0,              /* n_preallocs */
            (GInstanceInitFunc)text_undo_data_init,
            NULL
        };
        tu_type = g_type_register_static(G_TYPE_OBJECT, N_("TextUndoData"), &tu_info, 0);
    }
    return tu_type;
}


struct BufferModifier_;
typedef struct BufferModifier_  BufferModifier;

struct BufferModifier_ {
    void (*action)(GtkTextBuffer *buffer, struct BufferModifier_ *modifier);
    void (*destroy)(struct BufferModifier_ *modifier);
    gboolean partial :1;
    /* !partial are these list nodes which finish a composite operation
       (like "Replace All") and of course simple operations. */
};


static void buffer_modifier_destroy(BufferModifier *modifier)
{ }

typedef struct {
    BufferModifier base;
    gint start, len;
} BufferDeleter;

static void buffer_deleter_action(GtkTextBuffer *buffer, BufferModifier *modifier)
{
    BufferDeleter *deleter = (BufferDeleter*)modifier;
    GtkTextIter start, end;

    gtk_text_buffer_get_iter_at_offset(buffer, &start, deleter->start);
    end = start;
    gtk_text_iter_forward_chars(&end, deleter->len);
    gtk_text_buffer_delete(buffer, &start, &end);
}

static BufferModifier *buffer_deleter_new(const GtkTextIter *start, gint len)
{
    BufferDeleter *deleter;

    deleter = g_malloc(sizeof(BufferDeleter));
    deleter->base.action = buffer_deleter_action;
    deleter->base.destroy = buffer_modifier_destroy;
    deleter->base.partial = FALSE;
    deleter->start = gtk_text_iter_get_offset(start);
    deleter->len = len;
    return (BufferModifier*)deleter;
}

typedef struct {
    BufferModifier base;
    gint pos;
    gchar *text;
    /* Don't be tempted to use the char count here as for gtk_text_buffer_insert() is needed byte count. */
} BufferInserter;

static void buffer_inserter_destroy(BufferModifier *modifier)
{
    g_free(((BufferInserter*)modifier)->text);
}

static void buffer_inserter_action(GtkTextBuffer *buffer, BufferModifier *modifier)
{
    BufferInserter *inserter = (BufferInserter*)modifier;
    GtkTextIter iter;

    gtk_text_buffer_get_iter_at_offset(buffer, &iter, inserter->pos);
    gtk_text_buffer_insert(buffer, &iter, inserter->text, -1);
}

/* "Eats" text argument! */
static BufferModifier *buffer_inserter_new(const GtkTextIter *iter, gchar *text)
{
    BufferInserter *inserter;

    inserter = g_malloc(sizeof(BufferInserter));
    inserter->base.action = buffer_inserter_action;
    inserter->base.destroy = buffer_inserter_destroy;
    inserter->base.partial = FALSE;
    inserter->pos = gtk_text_iter_get_offset(iter);
    inserter->text = text;
    return (BufferModifier*)inserter;
}

/* TODO: add scroll position here */
typedef struct {
    BufferModifier base;
    gint cursor_pos;
    /*gboolean modified :1;*/
} BufferParameters;

static void buffer_parameters_action(GtkTextBuffer *buffer, BufferModifier *modifier)
{
    BufferParameters *params = (BufferParameters*)modifier;
    GtkTextIter iter;

    gtk_text_buffer_get_iter_at_offset(buffer, &iter, params->cursor_pos);
    gtk_text_buffer_place_cursor(buffer, &iter);
}

static BufferModifier *buffer_parameters_new(GtkTextBuffer *buffer)
{
    BufferParameters *params;
    GtkTextMark *cursor;
    GtkTextIter iter;

    params = g_malloc(sizeof(BufferParameters));
    params->base.action = buffer_parameters_action;
    params->base.destroy = buffer_modifier_destroy;
    params->base.partial = TRUE;
    cursor = gtk_text_buffer_get_insert(buffer);
    gtk_text_buffer_get_iter_at_mark(buffer, &iter, cursor);
    params->cursor_pos = gtk_text_iter_get_offset(&iter);
    return (BufferModifier*)params;
}

static void destroy_modifier(BufferModifier *modifier)
{
    modifier->destroy(modifier);
}

static GList *destroy_modifier_node(GList *list, GList *node)
{
    destroy_modifier((BufferModifier*)node->data);
    return g_list_delete_link(list, node);
}

/* In simle case remove the last element of the list. If it is a part of a complex action,
   may remove several elements. This function may be used to prevent out-of-memory.
   Always removes from undo (not redo) list! */
static void strip_modifications_list(TextUndoData *undo_data)
{
#if 0 /* FIXME: add back */
    GList *last, *pre_last;

    while(undo_data->undo/*undo_data->undo.length + undo_data->redo.length > max_undo_steps*/) {
        last = g_list_last(undo_data->undo);
        pre_last = g_list_previous(last);
        undo_data->undo = destroy_modifier_node(undo_data->undo, last);
        while(undo_data->undo) {
            last = pre_last;
            if( !((BufferModifier*)last->data)->partial ) break;
            pre_last = g_list_previous(last);
            undo_data->undo = destroy_modifier_node(undo_data->undo, last);
        }
    }
#endif /* 0 */
}

static void text_undo_data_abrupt(TextUndoData *undo_data)
{
    GList *node;

    for(node=undo_data->redo; node; node=g_list_next(node))
        destroy_modifier((BufferModifier*)node->data);
    g_list_free(undo_data->redo);
    undo_data->redo = NULL;
}

static void on_change_common_before(TextUndoData *undo_data)
{
    BufferModifier *state;

    if(!undo_data->now_undo_or_redo)
        text_undo_data_abrupt(undo_data); /* Notifications will be emitted in on_change_common_after. */
    state = buffer_parameters_new(undo_data->buffer);
    state->partial = TRUE; /* on_change_common_before() is always called before adding something to undo/redo list. */
    *undo_data->p_current = g_list_prepend(*undo_data->p_current, state);
}

static void text_undo_data_update_modified(TextUndoData *undo_data)
{
    if(undo_data->unmodified_point == 0)
        gtk_text_buffer_set_modified(undo_data->buffer, FALSE);
}

static void text_undo_data_notify(TextUndoData *undo_data)
{
    g_object_notify(G_OBJECT(undo_data), "can-undo");
    g_object_notify(G_OBJECT(undo_data), "can-redo");
}

static void on_change_common_after(TextUndoData *undo_data)
{
    /* See also on_end_user_action(). */
    /* For undo/redo operation unmodified_point is incremented/decremented in undo/redo functions. */
    if(!undo_data->now_user_action) {
        strip_modifications_list(undo_data);
        --undo_data->unmodified_point;
        if(undo_data->unmodified_point == 0)
            gtk_text_buffer_set_modified(undo_data->buffer, FALSE);
    }
    text_undo_data_notify(undo_data);
}

static void on_delete_range(GtkTextBuffer *buffer,
                            GtkTextIter *start,
                            GtkTextIter *end,
                            gpointer user_data)
{
    TextUndoData *undo_data = (TextUndoData*)user_data;
    BufferModifier *reverse;
    gchar *text;

    undo_data->user_action_non_empty = TRUE;
    on_change_common_before(undo_data);
    text = gtk_text_buffer_get_text(buffer, start, end, TRUE);
    reverse = buffer_inserter_new(start, text); /* Text is eaten. */
    reverse->partial = undo_data->now_user_action;
    *undo_data->p_current = g_list_prepend(*undo_data->p_current, reverse);
}

static void on_insert_text(GtkTextBuffer *buffer,
                           GtkTextIter *iter,
                           gchar *text, gint len,
                           gpointer user_data)
{
    TextUndoData *undo_data = (TextUndoData*)user_data;
    BufferModifier *reverse;

    undo_data->user_action_non_empty = TRUE;
    on_change_common_before(undo_data);
    reverse = buffer_deleter_new(iter, g_utf8_strlen(text, len));
    reverse->partial = undo_data->now_user_action;
    *undo_data->p_current = g_list_prepend(*undo_data->p_current, reverse);
}

static void on_delete_range_after(GtkTextBuffer *buffer,
                                  GtkTextIter *start,
                                  GtkTextIter *end,
                                  gpointer user_data)
{
    on_change_common_after((TextUndoData*)user_data);
}

static void on_insert_text_after(GtkTextBuffer *buffer,
                                 GtkTextIter *iter,
                                 gchar *text, gint len,
                                 gpointer user_data)
{
    on_change_common_after((TextUndoData*)user_data);
}

static void text_undo_data_erase(TextUndoData *undo_data)
{
    GList *node;

    for(node=undo_data->undo; node; node=g_list_next(node))
        destroy_modifier((BufferModifier*)node->data);
    g_list_free(undo_data->undo);
    for(node=undo_data->redo; node; node=g_list_next(node))
        destroy_modifier((BufferModifier*)node->data);
    g_list_free(undo_data->redo);
}

static void text_undo_data_reset(TextUndoData *undo_data)
{
    undo_data->undo = undo_data->redo = NULL;
    undo_data->now_user_action = FALSE;
    undo_data->now_undo_or_redo = FALSE;
    undo_data->unmodified_point = 0;
}

void text_undo_data_clear(TextUndoData *undo_data)
{
    text_undo_data_erase(undo_data);
    text_undo_data_reset(undo_data);
    text_undo_data_notify(undo_data);
}

static void on_begin_user_action(GtkTextBuffer *buffer, gpointer user_data)
{
    TextUndoData *undo_data = (TextUndoData*)user_data;

    undo_data->now_user_action = TRUE;
    undo_data->user_action_non_empty = FALSE;
}

static void on_end_user_action(GtkTextBuffer *buffer, gpointer user_data)
{
    TextUndoData *undo_data = (TextUndoData*)user_data;
    GList *item;

    undo_data->now_user_action = FALSE;
    if(!undo_data->user_action_non_empty) return;
    item = g_list_first(*undo_data->p_current);
    if(item) {
        ((BufferModifier*)item->data)->partial = FALSE;
        strip_modifications_list(undo_data);
    }
    if(!undo_data->now_undo_or_redo) {
        --undo_data->unmodified_point;
    }
    if(undo_data->unmodified_point == 0)
        gtk_text_buffer_set_modified(undo_data->buffer, FALSE);
}

static void on_modified_changed(GtkTextBuffer *textbuffer, gpointer user_data)
{
    TextUndoData *undo_data = (TextUndoData*)user_data;
    if(!gtk_text_buffer_get_modified(undo_data->buffer))
        undo_data->unmodified_point = 0;
}

TextUndoData *text_undo_data_new(GtkTextBuffer *buffer)
{
    return TEXT_UNDO_DATA( g_object_new(TYPE_TEXT_UNDO_DATA,
                                        N_("buffer"), buffer, NULL) );
}

static void text_undo_data_class_init(TextUndoDataClass *klass)
{
    GObjectClass *gobject_class;
    
    undo_data_quark = g_quark_from_static_string(N_("undo-data"));

    gobject_class = G_OBJECT_CLASS(klass);

    gobject_class->finalize = text_undo_data_finalize;
    gobject_class->set_property = text_undo_data_set_property;
    gobject_class->get_property = text_undo_data_get_property;
    klass->undo = text_undo_data_undo;
    klass->redo = text_undo_data_redo;

    g_object_class_install_property(gobject_class,
                                    PROP_BUFFER,
                                    g_param_spec_object(N_("buffer"),
                                                        _("Text buffer"),
                                                        _("Undo data is connected to this text buffer"),
                                                        GTK_TYPE_TEXT_BUFFER,
                                                        G_PARAM_READWRITE|G_PARAM_CONSTRUCT_ONLY));
    g_object_class_install_property(gobject_class,              
                                    PROP_CAN_UNDO,
                                    g_param_spec_boolean(N_("can-undo"),
                                                         _("Can undo"),
                                                         _("There are at least one undo step"),
                                                         FALSE,
                                                         G_PARAM_READABLE));
    g_object_class_install_property(gobject_class,
                                    PROP_CAN_REDO,
                                    g_param_spec_boolean(N_("can-redo"),
                                                         _("Can redo"),
                                                         _("There are at least one redo step"),
                                                         FALSE,
                                                         G_PARAM_READABLE));

    g_signal_new(N_("undo"),
                 TYPE_TEXT_UNDO_DATA,
                 G_SIGNAL_RUN_FIRST|G_SIGNAL_ACTION,
                 G_STRUCT_OFFSET(TextUndoDataClass, undo),
                 0, NULL,
                 g_cclosure_user_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
    g_signal_new(N_("redo"),
                 TYPE_TEXT_UNDO_DATA,
                 G_SIGNAL_RUN_FIRST|G_SIGNAL_ACTION,
                 G_STRUCT_OFFSET(TextUndoDataClass, redo),
                 0, NULL,
                 g_cclosure_user_marshal_VOID__VOID,
                 G_TYPE_NONE, 0);
    g_signal_new(N_("moved"),
                 TYPE_TEXT_UNDO_DATA,
                 G_SIGNAL_RUN_FIRST,
                 G_STRUCT_OFFSET(TextUndoDataClass, moved),
                 0, NULL,
                 g_cclosure_user_marshal_VOID__VOID/*OBJECT*/,
                 G_TYPE_NONE, 0);
}

static void text_undo_data_init(TextUndoData *undo_data)
{
    undo_data->buffer = NULL;
    undo_data->p_current = &undo_data->undo;
    undo_data->p_other   = &undo_data->redo;
    undo_data->max_undo_steps = -1;
    text_undo_data_reset(undo_data);
}

void text_undo_data_finalize(GObject *object)
{
    text_undo_data_erase(TEXT_UNDO_DATA(object));
}

static void text_undo_data_set_property(GObject      *object,
                                        guint         prop_id,
                                        const GValue *value,
                                        GParamSpec   *pspec)
{
    switch(prop_id) {
        case PROP_BUFFER: {
            GtkTextBuffer *buffer;
            TextUndoData *undo_data;

            undo_data = TEXT_UNDO_DATA(object);
            buffer = GTK_TEXT_BUFFER(g_value_get_object(value));
            undo_data->buffer = buffer;
            /* No need to add reference to buffer as undo data is always destroyed with buffer. */
            g_assert( !g_object_get_qdata(G_OBJECT(buffer), undo_data_quark) );
            g_object_set_qdata_full(G_OBJECT(buffer), undo_data_quark,
                                    object, (GDestroyNotify)g_object_unref);

            g_signal_connect(G_OBJECT(buffer), N_("begin-user-action"),
                             G_CALLBACK(on_begin_user_action), undo_data);
            g_signal_connect(G_OBJECT(buffer), N_("end-user-action"),
                             G_CALLBACK(on_end_user_action), undo_data);
            g_signal_connect(G_OBJECT(buffer), N_("modified-changed"),
                             G_CALLBACK(on_modified_changed), undo_data);
            g_signal_connect(G_OBJECT(buffer), N_("insert-text"),
                             G_CALLBACK(on_insert_text), undo_data);
            g_signal_connect(G_OBJECT(buffer), N_("delete-range"),
                             G_CALLBACK(on_delete_range), undo_data);
            g_signal_connect_data   (G_OBJECT(buffer), N_("insert-text"),
                                     G_CALLBACK(on_insert_text_after), undo_data,
                                     NULL, G_CONNECT_AFTER);
            g_signal_connect_data   (G_OBJECT(buffer), N_("delete-range"),
                                     G_CALLBACK(on_delete_range_after), undo_data,
                                     NULL, G_CONNECT_AFTER);
            break;
        }
        case PROP_CAN_UNDO:
        case PROP_CAN_REDO:
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void text_undo_data_get_property(GObject      *object,
                                        guint         prop_id,
                                        GValue       *value,
                                        GParamSpec   *pspec)
{
    switch(prop_id) {
        case PROP_BUFFER:
            g_value_set_object(value, TEXT_UNDO_DATA(object)->buffer);
            break;
        case PROP_CAN_UNDO:
            g_value_set_boolean(value, text_undo_data_can_undo(TEXT_UNDO_DATA(object)));
            break;
        case PROP_CAN_REDO:
            g_value_set_boolean(value, text_undo_data_can_redo(TEXT_UNDO_DATA(object)));
            break;
        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

gboolean text_undo_data_can_undo(TextUndoData *undo_data)
{
    /*g_assert( (undo_data->undo == NULL) == (undo_data->undo.length == 0) );*/
    /*g_assert( (undo_data->redo == NULL) == (undo_data->redo.length == 0) );*/
    return undo_data->undo != NULL;
}

gboolean text_undo_data_can_redo(TextUndoData *undo_data)
{
    /*g_assert( (undo_data->undo == NULL) == (undo_data->undo.length == 0) );*/
    /*g_assert( (undo_data->redo == NULL) == (undo_data->redo.length == 0) );*/
    return undo_data->redo != NULL;
}

static void text_undo_data_undo_or_redo(TextUndoData *undo_data)
{
    BufferModifier *modifier;
    GList *item;

    undo_data->now_undo_or_redo = TRUE;
    gtk_text_buffer_begin_user_action(undo_data->buffer);
    item = g_list_first(*undo_data->p_other);
    modifier = (BufferModifier*)item->data;
    for(;;) {
        (*modifier->action)(undo_data->buffer, modifier);
        *undo_data->p_other = g_list_delete_link(*undo_data->p_other, item);
        item = g_list_first(*undo_data->p_other);
        if(!item) break;
        modifier = (BufferModifier*)item->data;
        if(!modifier->partial) break;
    };
    gtk_text_buffer_end_user_action(undo_data->buffer);
    undo_data->now_undo_or_redo = FALSE;
}

void text_undo_data_undo(TextUndoData *undo_data)
{
    g_return_if_fail(text_undo_data_can_undo(undo_data));
    undo_data->p_current = &undo_data->redo;
    undo_data->p_other   = &undo_data->undo;
    text_undo_data_undo_or_redo(undo_data);
    undo_data->p_current = &undo_data->undo;
    undo_data->p_other   = &undo_data->redo;
    ++undo_data->unmodified_point;
    gtk_text_buffer_set_modified(undo_data->buffer, undo_data->unmodified_point != 0);

    /* I firstly call the notification as it is typically faster. */
    text_undo_data_notify(undo_data);
    g_signal_emit_by_name(undo_data, "moved");
}

void text_undo_data_redo(TextUndoData *undo_data)
{
    g_return_if_fail(text_undo_data_can_redo(undo_data));
    text_undo_data_undo_or_redo(undo_data);
    --undo_data->unmodified_point;
    gtk_text_buffer_set_modified(undo_data->buffer, undo_data->unmodified_point != 0);

    /* I firstly call the notification as it is typically faster. */
    text_undo_data_notify(undo_data);
    g_signal_emit_by_name(undo_data, "moved");
}

TextUndoData *my_text_buffer_get_undo_data(GtkTextBuffer *buffer)
{
    return g_object_get_qdata(G_OBJECT(buffer), undo_data_quark);
}

gboolean my_text_buffer_can_undo(GtkTextBuffer *buffer)
{
    return text_undo_data_can_undo(my_text_buffer_get_undo_data(buffer));
}

gboolean my_text_buffer_can_redo(GtkTextBuffer *buffer)
{
    return text_undo_data_can_redo(my_text_buffer_get_undo_data(buffer));
}

void my_text_buffer_undo(GtkTextBuffer *buffer)
{
    text_undo_data_undo(my_text_buffer_get_undo_data(buffer));
}

void my_text_buffer_redo(GtkTextBuffer *buffer)
{
    text_undo_data_redo(my_text_buffer_get_undo_data(buffer));
}
