/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "textstore.h"

#include <string.h>

#include "glade_support.h"
#include "main.h"
#include "encoding.h"

extern const int num_encodings_with_bom;

#ifdef G_OS_WIN32
const TextFileFormat default_text_file_format = { -1, N_("\r\n"), FALSE, FALSE };
#else
const TextFileFormat default_text_file_format = { -1, N_("\n"  ), FALSE, FALSE };
#endif /* G_OS_WIN32 */

static const gchar *const (BOMs[]) = {
    N_("\xEF\xBB\xBF"),     /* UTF-8 */
    N_("\xFE\xFF"),         /* UTF-16BE */
    N_("\xFF\xFE"),         /* UTF-16LE */
    N_("\x00\x00\xFE\xFF"), /* UTF-32BE */
    N_("\xFF\xFE\x00\x00")  /* UTF-32LE */
};

const gint BOM_lens[] = { 3, 2, 2, 4, 4 };

#define bom_max_chars 4

static gboolean io_channel_set_text_format(GIOChannel *channel,
                                           const TextFileFormat *format,
                                           GError **error)
{
    GIOStatus io_status;

    g_io_channel_set_line_term(channel,
                               format->line_term,
                               sizeof(format->line_term));
    io_status = g_io_channel_set_encoding(channel,
                                          encoding_names[format->encoding],
                                          error);
    return io_status == G_IO_STATUS_NORMAL;
}

/* TODO: Use temp file to do it atomically. */
gboolean text_buffer_save(GtkTextBuffer *buffer,
                          const gchar *file_name,
                          const TextFileFormat *format,
                          GError **error)
{
    GIOChannel *channel;
    GIOStatus io_status;
    gboolean ret;
    gint line_count, line;
    GtkTextIter iter, old_iter;
    gsize bytes_written;

    g_assert(format);

    channel = g_io_channel_new_file(file_name, "w", error);
    if(!channel) return FALSE;
    if(format->bom) {
        ret =  g_io_channel_write_chars(channel,
                                        BOMs[format->encoding], BOM_lens[format->encoding],
                                        &bytes_written,
                                        error);
        if(!ret) goto end_write;
    }
    ret = io_channel_set_text_format(channel, format, error);
    if(!ret) goto end_write;

    line_count = gtk_text_buffer_get_line_count(buffer);
    gtk_text_buffer_get_start_iter(buffer, &old_iter);
    for(line=1; line<=line_count; ++line) {
        gchar *str, *str2;
        gint str_size;
        gsize bytes_written;

        if(line == line_count)
            gtk_text_buffer_get_end_iter(buffer, &iter);
        else
            gtk_text_buffer_get_iter_at_line(buffer, &iter, line);

        str = gtk_text_buffer_get_text(buffer, &old_iter, &iter, TRUE);
        old_iter = iter;

        str_size = strlen(str); /* Probably, faster by iterator difference. */
        while(str_size>0) {
            switch(str[str_size-1]) {
                case '\n':
                case '\r':
                    break;
                default:
                    goto end_str_size;
                    break;
            }
            str[--str_size] = '\0';
        }
      end_str_size:

        str2 = g_strconcat(str, format->line_term, NULL);
        g_free(str);
        io_status = g_io_channel_write_chars(channel,
                                             str2, -1,
                                             &bytes_written,
                                             error);
        g_free(str2);
        if(io_status != G_IO_STATUS_NORMAL) {
            ret = FALSE;
            break;
        }
    }
  end_write:
    io_status = g_io_channel_shutdown(channel, TRUE,
                                      (error && *error) ? NULL : error);
    if(io_status != G_IO_STATUS_NORMAL) ret = FALSE;
    g_io_channel_unref(channel);
    return ret;
}

static gchar *get_file_text(const gchar                *file_name,
                            gsize                      *size,
                            TextFileFormat             *format,
                            text_file_format_callback   format_callback,
                            gpointer                    user_data,
                            gboolean                    ask_user,
                            GError                    **error)
{
    GIOChannel *channel;
    GIOStatus io_status;
    gshort guessed_enc;
    gboolean found_bom = FALSE;
    char bom_buf[bom_max_chars];
    gsize bytes_read;
    gchar *text;

    channel = g_io_channel_new_file(file_name, "r", error);
    if(!channel) return NULL;

    text = NULL;
    (void)g_io_channel_set_encoding(channel, NULL, NULL);
    io_status = g_io_channel_read_chars(channel,
                                        bom_buf, bom_max_chars, &bytes_read,
                                        error);
    format->bom = FALSE;
    format->bom_known = FALSE;
    switch(io_status) {
        case G_IO_STATUS_NORMAL:
            for(guessed_enc=0; guessed_enc<num_encodings_with_bom; ++guessed_enc) {
                found_bom = (gint)bytes_read >= BOM_lens[guessed_enc] &&
                            !strncmp(bom_buf, BOMs[guessed_enc], BOM_lens[guessed_enc]);
                if(found_bom) {
                    format->encoding = guessed_enc;
                    format->bom = TRUE;
                    format->bom_known = TRUE;
                    break;
                }
            }
            break;
        case G_IO_STATUS_EOF: /* File is empty. */
            return g_strdup("");
            break;
        default:
            goto end_read;
            break;
    }
    if( (!ask_user && format->encoding != -1 && format->line_term) ||
        (*format_callback)(format, user_data) )
    {
        gint8 bom_len;
        gsize term_len;

        term_len = strlen(format->line_term);
        bom_len = found_bom && format->encoding < num_encodings_with_bom ? BOM_lens[format->encoding] : 0;
        io_status = g_io_channel_seek_position(channel, bom_len, G_SEEK_SET, error);
        if(io_status != G_IO_STATUS_NORMAL) return NULL;
        io_status = g_io_channel_set_encoding(channel, encoding_names[format->encoding], error);
        if(io_status != G_IO_STATUS_NORMAL) return NULL;
        text = g_strdup("");
        *size = 0;
        g_io_channel_set_line_term(channel, NULL, 0);
        for(;;) {
            gchar *str;
            gsize terminator_pos;

            io_status = g_io_channel_read_line(channel, &str, NULL, &terminator_pos, error);
            if(io_status == G_IO_STATUS_EOF) break;
            if(io_status != G_IO_STATUS_NORMAL) {
                g_free(text);
                text = NULL;
                goto end_read;
            }
            text = g_realloc(text, *size+terminator_pos+1);
            memcpy(text+*size, str, terminator_pos);
            text[*size+terminator_pos] = '\n';
            *size += terminator_pos+1;
        }
    } else
        text = NULL;
  end_read:
    io_status = g_io_channel_shutdown(channel, TRUE,
                                      (error && *error) ? NULL : error);
    g_io_channel_unref(channel);
    return text;
}

gboolean text_buffer_load(GtkTextBuffer              *buffer,
                          const gchar                *file_name,
                          /*gsize                      *size,*/
                          TextFileFormat            **format,
                          text_file_format_callback   format_callback,
                          gpointer                    user_data,
                          gboolean                    ask_user,
                          GError                    **error)
{
    gchar *text;
    gsize size;

    if(!*format) *format = g_memdup(&default_text_file_format, sizeof default_text_file_format);
    text = get_file_text(file_name,
                         &size,
                         *format,
                         format_callback,
                         user_data,
                         ask_user,
                         error);
    if(text) {
        gtk_text_buffer_set_text(buffer, text, size);
        g_free(text);
        gtk_text_buffer_set_modified(buffer, FALSE);
        return TRUE;
    } else
        return FALSE;
}


gboolean text_buffer_insert_file_at_cursor(GtkWidget                 *window,
                                           GtkTextBuffer             *buffer,
                                           const gchar               *file_name,
                                           TextFileFormat            *format,
                                           text_file_format_callback  format_callback,
                                           gpointer                   user_data,
                                           gboolean                   ask_user)
{
    gchar *text;
    gsize size;
    GError *error = NULL;

    text = get_file_text(file_name,
                         &size,
                         format,
                         format_callback,
                         user_data,
                         ask_user,
                         &error);
    if(error) {
        my_g_error_message(window, error);
        g_free(text);
        return FALSE;
    }
    if(text) {
        gtk_text_buffer_insert_at_cursor(buffer, text, size);
        g_free(text);
        return TRUE;
    } else
        return FALSE;
}

/* TODO: inline */
char to_hex_digit(unsigned char digit)
{
    return digit < 10 ? digit+N_('0') : (digit-10)+N_('A');
}

unsigned char from_hex_digit(char digit)
{
    return digit <= N_('9') ? digit-N_('9') : (digit-N_('A'))+10;
}

/* Example: "0+UTF-8+0A"
   format ::= bom '+' encoding '+' line_term
   bom ::= '0' | '1'
   line_term ::= hex | line_term hex
   hex ::= hex_digit hex_digit
*/

const gchar *decode_line_term(const gchar *hex)
{
    gchar *line_term;
    
    /* For now accepted a limited set of line terms
       expressed with constant strings to not deallocate. */
    if( !strcmp(hex, N_("0D0A")) )
        line_term = N_("\r\n");
    else if( !strcmp(hex, N_("0A0D")) )
        line_term = N_("\n\r");
    else if( !strcmp(hex, N_("0D")) )
        line_term = N_("\r");
    else
        line_term = N_("\n");
    return line_term;
}

gchar *encode_line_term(const gchar *term)
{
    gchar *hex, *ptr;
    
    ptr = hex = g_malloc(strlen(term)*2 + 1);
    while(*term) {
        *ptr++ = to_hex_digit(*term/16);
        *ptr++ = to_hex_digit(*term%16);
        ++term;
    }
    *ptr = '\0';
    return hex;
}

gchar *text_file_format_to_str(const TextFileFormat *format)
{
    gchar *res, *term_hex;
    const gchar *ptr;

    ptr = format->line_term;
    term_hex = encode_line_term(format->line_term);
    res = g_strdup_printf(N_("%d+%s+%s"),
                          format->bom?1:0, encoding_names[format->encoding], term_hex);
    g_free(term_hex);
    return res;
}

TextFileFormat *text_file_format_from_str(const gchar *format_str)
{
    TextFileFormat *res;
    gchar *ptr;
    int i;
    
    res = (TextFileFormat*)g_memdup(&default_text_file_format, sizeof default_text_file_format);
    if(!*format_str) return res;
    res->bom = *format_str++ != N_('0');
    if(!*format_str) return res; /* sanity check */
    format_str++; /* skip '+' */
    if(!*format_str) return res; /* sanity check */
    ptr = strchr(format_str, N_('+'));
    if(!ptr) return res;
    res->encoding = 0/*UTF-8*/; /* fallback */
    for(i=0; i<num_encodings; ++i)
        if( !strncmp(format_str, encoding_names[i], ptr-format_str) ) {
            res->encoding = i;
            break;
        }
    format_str = ptr + 1;
    res->line_term = decode_line_term(format_str);
    return res;
}
