/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include "main.h"

#include <gdkconfig.h>

#include <stdlib.h>
#include <errno.h>
#ifdef HAVE_GNOME
#  include <libgnome/libgnome.h>
#endif
#ifdef G_OS_WIN32
#  include <windows.h> /* for is_windows_nt() */
#endif
#ifdef HAVE_GETOPT_H
#  include <getopt.h>
#endif

#include "gtk2edit16.xpm"
#include "gtk2edit32.xpm"
#include "gtk2edit48.xpm"

#include "glade_support.h"
#include "glade_interface.h"
#include "iniparser/iniparser.h"
#include "bindprops.h"
#include "mainwin.h"
#include "prefs.h"

GtkTooltips *file_tips;

gint the_line_height; /* in pixels */

#ifdef G_PLATFORM_WIN32
gboolean is_windows_nt(void)
{
    static int result = 2;
    OSVERSIONINFO info;

    if(result == 2) {
        info.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
        GetVersionEx(&info);
        result = info.dwPlatformId >= VER_PLATFORM_WIN32_NT;
    }
    return result != 0;
}
#endif /*G_PLATFORM_WIN32 */

static void error_message(GtkWidget *window, const gchar *text)
{
    GtkWidget *dialog;

    dialog = gtk_message_dialog_new(GTK_WINDOW(window),
                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
                                    text);
    my_disable_window_resize(GTK_WINDOW(dialog));
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
}

void errno_message(GtkWidget *window, const gchar *prefix)
{
    gchar *msg;
    gchar *text;
    gsize bytes_read, bytes_written;

    msg = g_locale_to_utf8(strerror(errno), -1,
                           &bytes_read, &bytes_written,
                           NULL);
    if(!msg) msg = g_strdup(_("Unknown error."));
    text = g_strdup_printf(N_("%s: %s"), prefix, msg);
    g_free(msg);
    error_message(window, text);
    g_free(text);
}

#ifdef GDK_WINDOWING_WIN32
void win32_error_message(GtkWidget *window, const gchar *prefix)
{
    gchar *msg, *msg_utf8, *text;
    GtkWidget *dialog;
    gsize bytes_written;

    msg = g_win32_error_message((gint)GetLastError());
    msg_utf8 = g_locale_to_utf8(msg, -1,
                                NULL, &bytes_written,
                                NULL);
    g_free(msg);
    if(!msg_utf8) msg_utf8 = g_strdup(_("Unknown error."));
    text = g_strdup_printf(N_("%s: %s"), prefix, msg_utf8);
    g_free(msg_utf8);
    dialog = gtk_message_dialog_new(GTK_WINDOW(window),
                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
                                    text);
    my_disable_window_resize(GTK_WINDOW(dialog));
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);
    g_free(text);
}
#endif /* GDK_WINDOWING_WIN32 */

void errno_or_win32_error_message(GtkWidget *window, const gchar *prefix)
{
#ifdef GDK_WINDOWING_WIN32
    win32_error_message(window, prefix);
#else
    errno_message(window, prefix);
#endif /* GDK_WINDOWING_WIN32 */
}

void my_g_error_message(GtkWidget *window, GError *error)
{
    error_message(window, error->message);
}

void alarm_message(GtkWidget *window, const char *str)
{
    gdk_beep(); /* TODO: Should also show message in statusbar */
}

#if defined(HAVE_GNOME) || defined(G_OS_WIN32)
void my_open_browser(const gchar *url)
{
#if defined(HAVE_GNOME)
    (void)gnome_url_show(url, NULL);
#elif defined(G_OS_WIN32)
    /* I don't bother to convert url into system locale. */
    (void)ShellExecute(NULL, TEXT("open"), url, NULL, NULL, SW_SHOWNORMAL);
#endif
}
#endif /* defined(HAVE_GNOME) || defined(G_PLATFORM_WIN32) */

static void on_about_dialog_response(GtkDialog *dialog,
                                     gint response,
                                     gpointer user_data)
{
    const gchar *const URLs[] = {
        N_("http://ex-code.com/gtk2edit/"),
        N_("http://ex-code.com/~porton/"),
        N_("http://www.regnow.com/softsell/nph-softsell.cgi?item=5790-4")
    };

    if(response <= 0) gtk_widget_destroy(GTK_WIDGET(dialog));
#if defined(HAVE_GNOME) || defined(G_OS_WIN32)
    if(response > 0) my_open_browser(URLs[response-1]);
#endif
}

void about_box(GtkWidget *mainwin)
{
    GtkWidget *dlg, *label;

#if defined(HAVE_GNOME) || defined(G_OS_WIN32)
    dlg = gtk_dialog_new_with_buttons(_("About program..."),
                                      GTK_WINDOW(mainwin),
                                      GTK_DIALOG_DESTROY_WITH_PARENT,
                                      _("App's _Homepage")   , 1,
                                      _("Au_thor's Homepage"), 2,
                                      _("_Purchase App")     , 3,
                                      GTK_STOCK_CLOSE        , GTK_RESPONSE_CANCEL,
                                      NULL);
#else
    dlg = gtk_dialog_new_with_buttons(_("About program..."),
                                      GTK_WINDOW(mainwin),
                                      GTK_DIALOG_DESTROY_WITH_PARENT,
                                      GTK_STOCK_CLOSE, GTK_RESPONSE_CANCEL,
                                      NULL);
#endif /* defined(HAVE_GNOME) || defined(G_OS_WIN32) */
    gtk_window_set_transient_for(GTK_WINDOW(dlg), GTK_WINDOW(mainwin));
    gtk_dialog_set_has_separator(GTK_DIALOG(dlg), FALSE);
    my_disable_window_resize(GTK_WINDOW(dlg));
    label = gtk_label_new(
        _("<big><b><span foreground='#ff0000'>GTK2 Text Editor</span></b></big> version " PACKAGE_VERSION
          " - <i>a Unicode text editor with many encodings</i>.\n"
          "<small><i>Copyright (c) 2003 Victor Porton. GPL 2.</i></small>\n\n"
          "<span color='#008000'>See "
          "<span foreground='#0000ff'>http://ex-code.com/gtk2edit/</span> "
          "for <b>supported version</b> (including if requested,\n"
          "<i>installer for Windows and CD; and no annoying about box at startup</i>).</span>\n\n"
          "Mail bug reports to <span foreground='#0000ff'>gtk2edit-devel@lists.sf.net</span>. Check "
          "the files README &amp; BUGS before reporting.\n\n"
          "I manage the following:\n"
          "- software development "
          "(including <b>development of <i>cheap</i> custom software</b> for you);\n"
          "- <b>Christian</b> ministry at "
          "<span foreground='#0000ff'>http://ex-code.com/~porton/bible/</span>.\n\n"
          "Please contact me (<span foreground='#0000ff'>porton@ex-code.com</span>) "
          "about these matters!\n\n"
          "<b><span color='red'>Please purchase this application. "
          "This will help free software development.</span></b>\n"));
    gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
    gtk_label_set_selectable(GTK_LABEL(label), TRUE);
    gtk_widget_show(label);
    gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(dlg)), 12);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dlg)->vbox), label);
    g_signal_connect(G_OBJECT(dlg), N_("response"),
                     G_CALLBACK(on_about_dialog_response), NULL);
    gtk_window_set_modal(GTK_WINDOW(dlg), TRUE);
    gtk_widget_show(dlg);
    /*gtk_dialog_run(GTK_DIALOG(dlg));*/
}

#ifdef LIMITED_VERSION
gboolean check_regcode(const gchar *full_name, guint32 code)
{
    const gchar *ptr;
    guint32 calc_code;

    calc_code = 0x7458E414;
    for(ptr=full_name; *ptr; ++ptr)
        calc_code = (calc_code << 3) ^ (calc_code >> 3) ^ 317*(guint8)*ptr;
    return calc_code == code;
}

gboolean is_registered_version()
{
    const gchar *name;
    guint32 code;

    name = preferences_get_user_name();
    code = preferences_get_user_code();
    return name ? check_regcode(name, code) : FALSE;
}
#endif /* LIMITED_VERSION */

static void check_first_run(GtkWidget *window)
{
#ifdef LIMITED_VERSION
    /* block */ {
        gboolean first_run;

        first_run = preferences_get_is_first_run();
        if(first_run || !is_registered_version()) about_box(window);
        if(first_run)
            preferences_set_is_first_run(FALSE);
    }
#else
    if(preferences_get_is_first_run()) {
        about_box(window);
        preferences_set_is_first_run(FALSE);
    }
#endif /* LIMITED_VERSION */
}

void call_text_font_dialog(GtkWidget *window)
{
    gchar *fontname;
    GtkWidget *selector;

    selector = gtk_font_selection_dialog_new(_("Select Text Font"));
    gtk_window_set_transient_for(GTK_WINDOW(selector), GTK_WINDOW(window));
    g_object_get(gtk_settings_get_default(), N_("text-font"), &fontname, NULL);
    if(fontname)
        gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(selector),
                                                fontname);
    g_free(fontname);
    if(gtk_dialog_run(GTK_DIALOG(selector)) == GTK_RESPONSE_OK) {
        fontname = gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(selector));
        g_object_set(gtk_settings_get_default(), N_("text-font"), fontname, NULL);
        g_free(fontname);
    }
    gtk_widget_destroy(GTK_WIDGET(selector));
}

void usage_msg(void)
{
    printf(_("gtk2edit [GTK-OPTIONS] [<FILE>]\n"));
}

int main(int argc, char **argv)
{
    GtkTooltips *tooltips;
    GtkWidget *mainwin;
    /*int option_index;*/
    const char *file_name_arg;
    const char *encoding;

    gtk_set_locale();

#ifdef ENABLE_NLS
    bindtextdomain(PACKAGE, LOCALEDIR);
    /*bind_textdomain_codeset(PACKAGE, N_("UTF-8"));*/
    textdomain(PACKAGE);
#endif

#ifdef G_PLATFORM_WIN32
    /* block */ {
        gchar *pkg_confdir, *conffile;

        pkg_confdir = g_win32_get_package_installation_subdirectory(
            N_("Extreme Code\\gtk2edit"), NULL, N_("etc"));
        conffile = g_strconcat(pkg_confdir, N_("\\gtk2edit.rc"), NULL);
        g_free(pkg_confdir);
        gtk_rc_add_default_file(conffile);
        g_free(conffile);
    }
#endif /*G_PLATFORM_WIN32 */

    gtk_rc_add_default_file(N_(CONF_DIR "/gtk2edit.rc"));

    gtk_init(&argc, &argv);

    /*option_index = 0;*/
    optind = 1;
    encoding = NULL;
#ifdef HAVE_GETOPT_LONG
    for(;;) {
        static const struct option long_options[] = {
            /* These options set a flag. */
            {N_("help"    ), no_argument      , NULL, 'h'},
            {N_("version" ), no_argument      , NULL, 'v'},
            /*{N_("encoding"), required_argument, NULL, 'e'},*/
            {0, 0, 0, 0}
        };
        int c;

        c = getopt_long(argc, argv, N_("h:"), long_options, NULL/*&option_index*/);
        if(c == -1) break;
        switch(c) {
            case 'h':
                usage_msg();
                printf(_("\nA simple convenient Unicode editor\n\n"
                         "Report bugs to <porton@ex-code.com> after reading file BUGS.\n"));
                exit(EXIT_SUCCESS);
                break;
            case 'v':
                printf(_("GTK2 Text Editor (gtk2edit) version " PACKAGE_VERSION "\n"
                         "(c) 2004 Extreme Code Software (http://www.ex-code.com).\n\n"
                         "Licensed under GPL2.\n\n"
                         "See http://ex-code.com/gtk2edit/ for updates.\n\n"
                         "VERY CHEAP custom software for you <porton@ex-code.com>\n"));
                exit(EXIT_SUCCESS);
                break;
#if 0
            case 'e':
                encoding = optarg;
                break;
#endif /* 0 */
        }
    }
#endif /* HAVE_GETOPT_LONG */

    if(argc - optind > 1) {
        usage_msg();
        exit(EXIT_FAILURE);
    }
    if(optind < argc)
        file_name_arg = argv[optind];
    else
        file_name_arg = NULL;

    g_set_prgname(_("GTK2 Text Editor"));

    preferences_init();

    /* block */ {
        GList *icon_list = 0;
        GError *error = 0;
        icon_list = g_list_append(icon_list,
                                  gdk_pixbuf_new_from_xpm_data((const char**)gtk2edit16_xpm));
        icon_list = g_list_append(icon_list,
                                  gdk_pixbuf_new_from_xpm_data((const char**)gtk2edit32_xpm));
        icon_list = g_list_append(icon_list,
                                  gdk_pixbuf_new_from_xpm_data((const char**)gtk2edit48_xpm));
        gtk_window_set_default_icon_list(icon_list);
    }

    file_tips = gtk_tooltips_new();

    mainwin = mainwin_new();

    check_first_run(mainwin);

    /* Placed after show for no warning messages. */
    if(file_name_arg) {
        gchar *file_name, *abs_path;
        MainwinData *win_data;
        gsize bytes_written;
        gpointer format;
        guint format_value;

        file_name = g_locale_to_utf8(file_name_arg, -1,
                                     NULL, &bytes_written,
                                     NULL); /* Error checking seems unnecessary, however... */
        win_data = mainwin_get_data(mainwin);
        /* I want absolute path in files history. */
        abs_path = file_name; /* if no assignment here, GCC's warning */
        if(!g_path_is_absolute(file_name)) {
            gchar *cur_dir;

            cur_dir = g_get_current_dir();
            abs_path = g_build_filename(cur_dir, file_name, NULL);
            g_free(cur_dir);
            g_free(file_name);
        }
        g_object_get(gtk_settings_get_default(),
                     N_("always-ask-encoding"), &format_value,
                     NULL);
        format = format_value != 0 ? NULL
                                   : file_document_get_format_copy(win_data->file_document);
        file_document_do_load_file(win_data->file_document, abs_path, &format);
        g_free(abs_path);
    }

    gtk_main();

    return EXIT_SUCCESS;
}

/* TODO: Move these functions. */
static void on_disable_resize_window_realize(GtkWidget *widget, gpointer user_data)
{
    gdk_window_set_decorations(widget->window,
                               (GdkWMDecoration)(GDK_DECOR_TITLE|GDK_DECOR_BORDER|GDK_DECOR_MENU));
}

void my_disable_window_resize(GtkWindow *window)
{
    gtk_window_set_resizable(window, FALSE);
    g_signal_connect(G_OBJECT(window), N_("realize"),
                     G_CALLBACK(on_disable_resize_window_realize), NULL);
}

GtkWidget *my_gtk_widget_get_toplevel(GtkWidget *widget)
{
    GtkWidget *win = NULL;

    for(;;) {
        widget = GTK_IS_MENU(widget) ? gtk_menu_get_attach_widget(GTK_MENU(widget))
                                     : gtk_widget_get_parent(widget);
        if(!widget) break;
        win = widget;
    }
    return win;
}
