/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "glade_support.h"

#include "encoding.h"

#include <string.h>

static void on_encoding_selection_changed(GtkTreeSelection *selection, gpointer data)
{
    SelectEncodingDialog *se_dlg = (SelectEncodingDialog*)data;
    GtkTreeIter iter;
    gboolean enable;
    GtkTreeModel *model;

    if(!gtk_tree_selection_get_selected(selection, &model, &iter))
        enable = FALSE;
    else {
        GValue value;
        gint num;

        memset(&value, 0, sizeof(value));
        gtk_tree_model_get_value(model, &iter, 1, &value);
        num = g_value_get_int(&value); /* A separate assignment to prevent GCC warning */
        enable = num < num_encodings_with_bom;
        g_value_unset(&value);
    }
    if(se_dlg->bom_check) gtk_widget_set_sensitive(se_dlg->bom_check, enable);
}

static void on_encoding_row_activate(GtkTreeView *treeview,
                                     GtkTreePath *path,
                                     GtkTreeViewColumn *arg2,
                                     gpointer user_data)
{
    SelectEncodingDialog *se_dlg = (SelectEncodingDialog*)user_data;

    gtk_dialog_response(GTK_DIALOG(se_dlg->dialog), GTK_RESPONSE_ACCEPT);
}

void create_select_encoding_dialog(GtkWidget *mainwin,
                                   const gchar *title,
                                   SelectEncodingDialog *se_dlg,
                                   gboolean is_write,
                                   TextFileFormat *fmt)
{
    GtkTreeSelection *select;
    GtkListStore *store;
    GtkWidget *box, *scroll;
    gint cur;

    if(fmt->encoding == -1) {
        fmt->encoding = 0;
#ifdef G_OS_WIN32
        fmt->line_term = "\r\n";
#else
        fmt->line_term = "\n";
#endif /* G_OS_WIN32 */
    }
    fmt->bom = !enc_has_bom(fmt); /* Turn on BOM by default. */

    se_dlg->dialog = gtk_dialog_new_with_buttons(
        title,
        GTK_WINDOW(mainwin),
        GTK_DIALOG_MODAL|GTK_DIALOG_DESTROY_WITH_PARENT,
        GTK_STOCK_OK,
        GTK_RESPONSE_ACCEPT,
        GTK_STOCK_CANCEL,
        GTK_RESPONSE_CANCEL,
        NULL);
    gtk_dialog_set_default_response(GTK_DIALOG(se_dlg->dialog), GTK_RESPONSE_ACCEPT);
    gtk_window_set_default_size(GTK_WINDOW(se_dlg->dialog), -1, 500);

    store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_INT);
    se_dlg->list_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(se_dlg->list_view), FALSE);
    gtk_tree_view_set_enable_search(GTK_TREE_VIEW(se_dlg->list_view), TRUE);
    select = gtk_tree_view_get_selection(GTK_TREE_VIEW(se_dlg->list_view));
    gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
    gtk_tree_view_append_column(
        GTK_TREE_VIEW(se_dlg->list_view),
        gtk_tree_view_column_new_with_attributes(_("Encoding"),
                                                 gtk_cell_renderer_text_new(),
                                                 N_("text"), 0,
                                                 NULL));
    for(cur=0; cur<num_encodings; ++cur) {
        GtkTreeIter iter;

        gtk_list_store_append(store, &iter);
        gtk_list_store_set(store, &iter, 0, encoding_names[cur], 1, (gint)cur, -1);
    }
    /* block */ {
        GtkTreePath *path;

        path = gtk_tree_path_new();
        gtk_tree_path_append_index(path, (gint)(fmt->encoding));
#if 0
        gtk_tree_selection_select_path(
            gtk_tree_view_get_selection(GTK_TREE_VIEW(se_dlg->list_view)),
            path);
#endif /* 0 */
        gtk_tree_view_set_cursor(GTK_TREE_VIEW(se_dlg->list_view), path, NULL, FALSE);
        gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(se_dlg->list_view),
                                     path, NULL,
                                     FALSE, 0.0f, 0.0f);
        gtk_tree_path_free(path);
    }
    box = GTK_DIALOG(se_dlg->dialog)->vbox;
    scroll = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
                                   GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll), GTK_SHADOW_NONE);
    gtk_container_add(GTK_CONTAINER(scroll), se_dlg->list_view);
    gtk_widget_show_all(scroll);
    gtk_box_pack_start(GTK_BOX(box), scroll, TRUE, TRUE, 2);
    gtk_dialog_set_default_response(GTK_DIALOG(se_dlg->dialog), GTK_RESPONSE_ACCEPT);
    se_dlg->bom_check = se_dlg->crlf_check = NULL;
    g_signal_connect_after(G_OBJECT(se_dlg->list_view), N_("row-activated"),
                             G_CALLBACK(on_encoding_row_activate), se_dlg);
    if(is_write) {
        se_dlg->bom_check  = gtk_check_button_new_with_label(_("Add encoding mark at start of file"));
        se_dlg->crlf_check = gtk_check_button_new_with_label(_("Use Windows line separators (CRLF)"));
        gtk_widget_set_sensitive(se_dlg->bom_check, enc_has_bom(fmt));
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(se_dlg->bom_check),
                                     fmt->bom);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(se_dlg->crlf_check),
                                     fmt->line_term[0] == '\r');
        gtk_widget_show(se_dlg->bom_check);
        gtk_widget_show(se_dlg->crlf_check);
        gtk_box_pack_start(GTK_BOX(box), se_dlg->bom_check, FALSE, FALSE, 2);
        gtk_box_pack_start(GTK_BOX(box), se_dlg->crlf_check, FALSE, FALSE, 2);
        g_signal_connect(G_OBJECT(select), N_("changed"),
                         G_CALLBACK(on_encoding_selection_changed), se_dlg);
    }
}

void select_encoding_dialog_get_encoding(SelectEncodingDialog *se_dlg,
                                         TextFileFormat *fmt)
{
    GtkTreeModel *model;
    GtkTreeIter iter;
    GtkTreeSelection *selection;
    gboolean crlf;

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(se_dlg->list_view));
    if(gtk_tree_selection_get_selected(selection, &model, &iter))
    {
        GValue value;
        gint num;

        memset(&value, 0, sizeof(value));
        gtk_tree_model_get_value(model, &iter, 1, &value);
        /* A separate assignment to prevent GCC warning */
        fmt->encoding = g_value_get_int(&value);
        g_value_unset(&value);
        fmt->bom = se_dlg->bom_check && enc_has_bom(fmt) &&
                  gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(se_dlg->bom_check));
#ifdef G_OS_WIN32
        crlf = !se_dlg->crlf_check ||
            gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(se_dlg->crlf_check));
#else
        crlf = se_dlg->crlf_check &&
            gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(se_dlg->crlf_check));
#endif /* G_OS_WIN32 */
        fmt->line_term = crlf ? "\r\n" : "\n";
    } else
        *fmt = default_text_file_format;
}

gboolean enc_has_bom(const TextFileFormat *fmt)
{
    return fmt->encoding < num_encodings_with_bom;
}
