/*
** 1998-05-29 -	This command is simply essential. It selects all entries in the current
**		pane.
** 1998-06-04 -	Greatly expanded; now also includes commands to deselect all entries,
**		toggle selected/unselected, and a quite powerful regular expression
**		selection tool. :) Also added freezing/thawing of the dirpane clist
**		during execution of these routines; helps speed, improves looks.
** 1998-06-05 -	Added a glob->RE translator, triggered by a check box in the selreg
**		dialog. Simplifies life for casual users (like myself). Also added
**		a check box which allows the user to invert the RE matching set, i.e.
**		to act upon all entries that do NOT match the expression.
**		BUG: The matching should check for regexec() internal errors!
** 1998-08-02 -	Added (mysteriously missing) support for SM_TYPE_FILES in selection,
**		and also closing the dialog by return/enter (accepts) and escape (cancels).
** 1998-09-19 -	Replaced regexp routines with the POSIX ones in the C library, which I
**		just discovered. Better.
** 1999-03-04 -	Changes in how selections are handled through-out gentoo made some
**		changes in this module necessary. We're on GTK+ 1.2.0 now.
** 1999-05-06 -	Adapted for built-in command argument support. Really cool.
** 1999-06-19 -	Adapted for the new dialog module. Fun, fun, fun. Not. ;^)
** 1999-11-14 -	Added support for user-selectable content matching in SelectRE. I like it.
*/

#include "gentoo.h"

#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <regex.h>

#include <gdk/gdkkeysyms.h>

#include "cmdparse.h"
#include "dirpane.h"
#include "dialog.h"
#include "dpformat.h"
#include "events.h"
#include "fileutil.h"
#include "guiutil.h"
#include "miscutil.h"
#include "strutil.h"
#include "types.h"

#include "cmd_select.h"

/* ----------------------------------------------------------------------------------------- */

#define	SM_SET			(0)
#define	SM_SET_ALL		(0)
#define	SM_SET_SELECTED		(1)
#define	SM_SET_UNSELECTED	(2)

#define	SM_TYPE			(1)
#define	SM_TYPE_ALL		(0)
#define	SM_TYPE_DIRS		(1)
#define	SM_TYPE_FILES		(2)

#define	SM_ACTION		(2)
#define	SM_ACTION_SELECT	(0)
#define	SM_ACTION_UNSELECT	(1)
#define	SM_ACTION_TOGGLE	(2)

typedef void (*SelAction)(DirPane *, gint);

/* Base selection parameters. Filled-in either by GUI, or by direct command arguments. */
typedef struct {
	guint		set;		/* In which set of rows should we operate? */
	guint		type;		/* Do we operate only on files, dirs, or both? */
	guint		action;		/* What do we do with matching rows? */
	DPContent	content;	/* Which field of the pane should we match against? */
} SelParam;

typedef struct {
	SelParam	basic;		/* Basic parameters (selection set, action, content). */
	gboolean	glob;		/* Treat the RE as a glob pattern? */
	gboolean	invert;		/* Invert the matching sense? */
	gboolean	full;		/* Require RE to match entire row? */
	gboolean	nocase;		/* Ignore case differences? */
	GString		*re;		/* The actual regular expression string. */
} SelParamRE;

typedef struct {
	SelParam	basic;		/* Basic parameters (selection set, action, content). */
	GString		*cmd;		/* Shell command to execute. */
} SelParamShell;

static SelParamRE old_sp_re = {
	{ SM_SET_ALL, SM_TYPE_ALL, SM_ACTION_SELECT, DPC_NAME },
	TRUE, FALSE, TRUE, FALSE, NULL 
};

static SelParamShell old_sp_shell = {
	{ SM_SET_ALL, SM_TYPE_ALL, SM_ACTION_SELECT, DPC_NAME },
	NULL
};
				

/* ----------------------------------------------------------------------------------------- */

/* Select a specific row, or, if the row=ROW keyword is not specified, the one that was last clicked. */
gint cmd_selectrow(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	GdkEventButton	*evt;
	const gchar	*num;
	gint		row = -1;

	if((num = car_keyword_get_value(ca, "row", NULL)) != NULL)
	{
		gchar	*endptr;

		row = strtol(num, &endptr, 0);
		if(!*num || *endptr)
			row = -1;
	}
	else if((evt = (GdkEventButton *) evt_event_get(GDK_BUTTON_PRESS)) != NULL)
	{
		gint	col;

		if(!gtk_clist_get_selection_info(GTK_CLIST(src->list), evt->x, evt->y, &row, &col) || (row > src->dir.num_rows))
			row = -1;
	}

	if(row != -1)
	{
		guint	action;
		void	(*afunc)(DirPane *dp, gint row);

		action = car_keyword_get_enum(ca, "action", 0, "select", "unselect", "toggle", NULL);
		afunc = (action == 0) ? dp_select : (action == 1) ? dp_unselect : dp_toggle;
		afunc(src, row);
		gui_events_flush();
	}
	return 1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-04 -	Select all rows in the currently active pane. Rewritten for the new cooler
**		selection management made possible by GTK+ 1.2.0.
*/
gint cmd_selectall(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	dp_select_all(src);
	return 1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-03-04 -	Unselect all rows of current pane. Rewritten. */
gint cmd_selectnone(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	dp_unselect_all(src);
	return 1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-06-04 -	Invert the selection set, making all currently selected files become
**		unselected, and all unselected become selected.
** 2002-08-25 -	Complete rewrite, now uses bit vector for speed. 40-100X faster.
*/
gint cmd_selecttoggle(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	guint	*bm, word, row, mask;
	gsize	alloc;
	GSList	*slist;

	/* Compute size of, and allocate selection bit vector. */
	alloc = (src->dir.num_rows + CHAR_BIT * sizeof *bm - 1) / (CHAR_BIT * sizeof *bm);
	bm = g_malloc(alloc * sizeof *bm);
	memset(bm, 0, alloc * sizeof *bm);	/* Assume no selection. */

	gtk_clist_freeze(src->list);

	/* Was there a selection? If so, set bits for selected rows. */
	if((slist = dp_get_selection(src)) != NULL)
	{
		const GSList	*iter;

		for(iter = slist; iter != NULL; iter = g_slist_next(iter))
		{
			register guint	n;

			n = DP_ROW_INDEX(src, DP_SEL_ROW(iter));
			bm[n / (CHAR_BIT * sizeof *bm)] |= 1 << (n % (CHAR_BIT * sizeof *bm));
		}
		dp_free_selection(slist);
	}
	/* Now invert the selection bitmap. */
	for(word = 0; word < alloc; word++)
		bm[word] = ~bm[word];
	/* Clear selection, and apply the (inverted) bitmap. */
	dp_unselect_all(src);
	for(word = row = 0; row < src->dir.num_rows; word++)
	{
		if(!bm[word])		/* Test a bunch (typically 32) rows in parallel. */
		{
			row += CHAR_BIT * sizeof *bm;
			continue;
		}
		for(mask = 1; mask; mask <<= 1, row++)
		{
			if(bm[word] & mask)
				dp_select(src, row);
		}
	}
	gtk_clist_thaw(src->list);
	g_free(bm);

	return 1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-05-06 -	Inspect which set (selected or unselected) the given row is, and determine
**		if it complies with the wanted one as specified in <sp>.
*/
static gboolean filter_set(DirPane *dp, guint row, SelParam *sp)
{
	gboolean	sel;

	if(sp->set == SM_SET_ALL)
		return TRUE;
	sel = dp_is_selected(dp, &dp->dir.row[row]);
	return (sp->set == SM_SET_SELECTED) ? sel : !sel;
}

/* 1999-05-06 -	Inspect type (dir or non-dir) given row has, and match against <sp>. */
static gboolean filter_type(DirPane *dp, guint row, SelParam *sp)
{
	gboolean	dir;

	if(sp->type == SM_TYPE_ALL)
		return TRUE;
	dir = S_ISDIR(DP_ROW_LSTAT(&dp->dir.row[row]).st_mode);

	return (sp->type == SM_TYPE_DIRS) ? dir : !dir;
}

/* 1999-11-14 -	Compute the column of given pane's CList to query to get the wanted data.
**		The answer is either a column in the 0..(num_columns-1) range, or -1 for
**		failure.
*/
static guint compute_column(DirPane *dp, DPContent content)
{
	MainInfo	*min = dp->main;
	DPFormat	*dpf = &min->cfg.dp_format[dp->index];
	guint		i;

	for(i = 0; i < dpf->num_columns; i++)
	{
		if(dpf->format[i].content == content)
			return i;
	}
	return 0;
}

/* 2005-06-04 -	Select action fuction. The <action> should be one of the SM_ACTION_ values. */
static SelAction select_action_func(gint action)
{
	switch(action)
	{
		case SM_ACTION_SELECT:
			return dp_select;
		case SM_ACTION_UNSELECT:
			return dp_unselect;
		case SM_ACTION_TOGGLE:
			return dp_toggle;
	}
	return NULL;
}

/* 1999-05-06 -	Once we have all the selection's parameters neatly tucked into <sp>, perform the
**		actual operation.
*/
static void do_selectre(MainInfo *min, DirPane *dp, SelParamRE *sp)
{
	GString		*tmp;
	regex_t		re;
	void		(*sel_action)(DirPane *dp, gint row) = NULL;

	if(sp->re == NULL)
		return;
	sel_action = select_action_func(sp->basic.action);
	if((tmp = g_string_new(sp->re->str)) != NULL)
	{
		gint	reret;

		if(sp->glob)
			stu_gstring_glob_to_re(tmp);
		if(sp->full)	/* Require RE to match entire name? Then surround it with "^$". */
		{
			g_string_prepend_c(tmp, '^');
			g_string_append_c(tmp,  '$');
		}
		if((reret = regcomp(&re, tmp->str, REG_EXTENDED | (sp->nocase ? REG_ICASE : 0))) == 0)
		{
			gchar	*text;
			gint	matchok;
			guint	i, column;

			dp_freeze(dp);
			matchok = sp->invert ? REG_NOMATCH : 0;
			column  = compute_column(dp, sp->basic.content);
			for(i = 0; i < dp->dir.num_rows; i++)
			{
				if(filter_set(dp, i, &sp->basic) && filter_type(dp, i, &sp->basic))
				{
					gtk_clist_get_text(GTK_CLIST(dp->list), i, column, &text);
					if(regexec(&re, text, 0, NULL, 0) == matchok)
						sel_action(dp, i);
				}
			}
			dp_thaw(dp);
			regfree(&re);
		}
		else
		{
			gchar	ebuf[1024];

			regerror(reret, &re, ebuf, sizeof ebuf);
			dlg_dialog_async_new_error(ebuf);
		}
		g_string_free(tmp, TRUE);
	}
}

/* 1999-05-06 -	Build a mode selection menu, using items from <def>. Items are separated by vertical
**		bars. Each item will have it's user data set to the index of the item in question.
*/
static GtkWidget * build_mode_menu(const gchar *def)
{
	GtkWidget	*menu = NULL, *item;
	gchar		*temp, *iname, *iend;
	guint		minor;

	if((temp = g_strdup(def)) != NULL)
	{
		menu = gtk_menu_new();
		for(iname = temp, minor = 0; (*iname != '\0') && (iend = strchr(iname, '|')) != NULL; iname = iend + 1, minor++)
		{
			*iend = '\0';
			item = gtk_menu_item_new_with_label(iname);
			gtk_menu_append(GTK_MENU(menu), item);
			gtk_object_set_user_data(GTK_OBJECT(item), GUINT_TO_POINTER(minor));
		}
		g_free(temp);
	}
	return menu;
}

/* 1999-11-14 -	Build a menu with one choice per active content type in the given pane. */
static GtkWidget * build_content_menu(DirPane *dp)
{
	MainInfo	*min = dp->main;
	DPFormat	*dpf;
	GtkWidget	*menu, *item;
	guint		i;

	dpf = &min->cfg.dp_format[dp->index];
	menu = gtk_menu_new();
	for(i = 0; i < dpf->num_columns; i++)
	{
		if(dpf->format[i].content == DPC_ICON)
			continue;
		item = gtk_menu_item_new_with_label(dpf_get_content_name(dpf->format[i].content));
		gtk_object_set_user_data(GTK_OBJECT(item), GUINT_TO_POINTER(dpf->format[i].content));
		gtk_menu_append(GTK_MENU(menu), item);
	}
	return menu;
}

/* 2005-06-04 -	Compute which index in a content menu (as created by build_content_menu(), above)
 * 		a given content type has. Returns 0 if <content> is not used by <dp>.
*/
static guint content_menu_find(const DirPane *dp, DPContent content)
{
	const MainInfo	*min = dp->main;
	const DPFormat	*dpf;
	guint		i, index;

	dpf = &min->cfg.dp_format[dp->index];
	for(i = index = 0; i < dpf->num_columns; i++)
	{
		if(dpf->format[i].content == DPC_ICON)
			continue;
		if(dpf->format[i].content == content)
			return index;
		index++;
	}
	return 0;
}

/* 1999-05-06 -	Build option menus for mode selections. */
static void selparam_init_menus(DirPane *dp, GtkWidget *vbox, GtkWidget *opt_menu[4], guint *history[4])
{
	const gchar	*mdef[] = { N_("All rows|Selected|Unselected|"),
				  N_("All types|Directories only|Non-directories only|"),
				  N_("Select|Unselect|Toggle|") },
			*mlabel[] = { N_("Set"), N_("Type"), N_("Action") };
	GtkWidget	*table, *label, *menu;
	guint		i;

	table = gtk_table_new(3, 3, FALSE);
	for(i = 0; i < sizeof mdef / sizeof mdef[0]; i++)
	{
		label = gtk_label_new(_(mlabel[i]));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, i, i + 1, 0,0,0,0);
		menu = build_mode_menu(_(mdef[i]));
		opt_menu[i] = gtk_option_menu_new();
		gtk_option_menu_set_menu(GTK_OPTION_MENU(opt_menu[i]), menu);
		gtk_option_menu_set_history(GTK_OPTION_MENU(opt_menu[i]), *history[i]);
		gtk_table_attach(GTK_TABLE(table), opt_menu[i], 1, 2, i, i+1, GTK_EXPAND|GTK_FILL,0,0,0);
	}
	label = gtk_label_new(_("Content"));
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, i, i + 1, 0,0,0,0);
	menu = build_content_menu(dp);
	opt_menu[3] = gtk_option_menu_new();
	gtk_option_menu_set_menu(GTK_OPTION_MENU(opt_menu[3]), menu);
	gtk_option_menu_set_history(GTK_OPTION_MENU(opt_menu[3]), content_menu_find(dp, *history[3]));
	gtk_table_attach(GTK_TABLE(table), opt_menu[3], 1, 2, i, i + 1, GTK_EXPAND|GTK_FILL,0,0,0);

	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);
}

/* 2005-06-04 -	Parse out basic select parameters from command arguments. */
static void selparam_set_from_cmdarg(SelParam *sp, const CmdArg *ca)
{
	const gchar	*content;

	/* Always allow non-bareword arguments to alter the settings. */
	sp->set    = car_keyword_get_enum(ca, "set", sp->set, "all rows", "selected", "unselected", NULL);
	sp->type   = car_keyword_get_enum(ca, "type", sp->type, "all types", "directories only", "non-directories only", NULL);
	sp->action = car_keyword_get_enum(ca, "action", sp->action, "select", "unselect", "toggle", NULL);
	if((content = car_keyword_get_value(ca, "content", NULL)) != NULL)
	{
		DPContent	ac;

		if((ac = dpf_get_content_from_mnemonic(content)) < DPC_NUM_TYPES)
			sp->content = ac;
	}
}

/* 2005-06-04 -	Get pointers to each value in a SelParam, useful with option menus and selparam_init_menus(). */
static void selparam_get_value_pointers(guint **ptr, SelParam *sp)
{
	ptr[0] = &sp->set;
	ptr[1] = &sp->type;
	ptr[2] = &sp->action;
	ptr[3] = &sp->content;
}

static void selparam_read_menus(guint **ptr, GtkWidget *opt_menu[])
{
	GtkWidget	*menu, *mi;
	guint		i;

	for(i = 0; i < 4; i++)
	{
		if((menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(opt_menu[i]))) != NULL)
		{
			if((mi = gtk_menu_get_active(GTK_MENU(menu))) != NULL)
				*ptr[i] = GPOINTER_TO_UINT(gtk_object_get_user_data(GTK_OBJECT(mi)));
		}
	}
}

/* 1998-06-04 -	Pop up a dialog window asking the user for a regular expression, and
**		then match only those entries whose names match. Very Opus.
** 1999-05-06 -	Basically rewritten, to accomodate command arguments in a nice way. Now
**		the GUI uses no external state; all values are extracted from the
**		widgets after the dialog is closed.
*/
gint cmd_selectre(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	SelParamRE	sp;
	const gchar	*str;
	guint		res = 0;

	/* Always start from the previous set of parameters. */
	sp = old_sp_re;

	selparam_set_from_cmdarg(&sp.basic, ca);
	sp.glob   = car_keyword_get_boolean(ca, "glob", sp.glob);
	sp.invert = car_keyword_get_boolean(ca, "invert", sp.invert);
	sp.full   = car_keyword_get_boolean(ca, "full", sp.full);
	sp.nocase = car_keyword_get_boolean(ca, "nocase", sp.nocase);

	/* If actual bareword expression found, it's all we need to do a search. */
	if((str = car_bareword_get(ca, 0)) != NULL)
	{
		if(sp.re == NULL)
			sp.re = g_string_new(str);
		else
			g_string_assign(sp.re, str);
	}	
	else			/* No bareword, so pop up dialog with settings, asking for expression. */
	{
		GtkWidget	*vbox, *opt_menu[4], *glob, *invert, *full, *nocase, *re;
		guint		*mptr[sizeof opt_menu / sizeof *opt_menu];
		Dialog		*dlg;

		mptr[0] = &sp.basic.set;
		mptr[1] = &sp.basic.type;
		mptr[2] = &sp.basic.action;
		mptr[3] = &sp.basic.content;

		vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_set_border_width(GTK_CONTAINER(vbox), 2);
		selparam_init_menus(src, vbox, opt_menu, mptr);

		glob = gtk_check_button_new_with_label(_("Treat RE as Glob Pattern?"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glob), sp.glob);
		gtk_box_pack_start(GTK_BOX(vbox), glob, FALSE, FALSE, 0);
		invert = gtk_check_button_new_with_label(_("Invert RE Matching?"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(invert), sp.invert);
		gtk_box_pack_start(GTK_BOX(vbox), invert, FALSE, FALSE, 0);
		full = gtk_check_button_new_with_label(_("Require Match on Full Name?"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(full), sp.full);
		gtk_box_pack_start(GTK_BOX(vbox), full, FALSE, FALSE, 0);
		nocase = gtk_check_button_new_with_label(_("Ignore Case?"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(nocase), sp.nocase);
		gtk_box_pack_start(GTK_BOX(vbox), nocase, FALSE, FALSE, 0);

		re = gtk_entry_new();
		if(sp.re != NULL)
		{
			gtk_entry_set_text(GTK_ENTRY(re), sp.re->str);
			gtk_entry_select_region(GTK_ENTRY(re), 0, -1);
		}
		gtk_box_pack_start(GTK_BOX(vbox), re, FALSE, FALSE, 0);
		dlg = dlg_dialog_sync_new(vbox, _("Select Using RE"), NULL);
		gtk_widget_grab_focus(re);
		res = dlg_dialog_sync_wait(dlg);
		if(res == DLG_POSITIVE)		/* If user accepted, collect widget state into 'sp'. */
		{
			GtkWidget	*menu, *mi;
			guint		i;

			for(i = 0; i < 3; i++)
			{
				if((menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(opt_menu[i]))) != NULL)
				{
					if((mi = gtk_menu_get_active(GTK_MENU(menu))) != NULL)
						*mptr[i] = GPOINTER_TO_UINT(gtk_object_get_user_data(GTK_OBJECT(mi)));
				}
			}
			menu = gtk_option_menu_get_menu(GTK_OPTION_MENU(opt_menu[3]));
			mi   = gtk_menu_get_active(GTK_MENU(menu));
			sp.basic.content = (DPContent) GPOINTER_TO_UINT(gtk_object_get_user_data(GTK_OBJECT(mi)));
			sp.glob	  = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(glob));
			sp.invert = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(invert));
			sp.full   = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(full));
			sp.nocase = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(nocase));
			if(sp.re != NULL)
				g_string_assign(sp.re, gtk_entry_get_text(GTK_ENTRY(re)));
			else
				sp.re = g_string_new(gtk_entry_get_text(GTK_ENTRY(re)));
		}
		dlg_dialog_sync_destroy(dlg);
	}
	if(res == 0)
	{
		do_selectre(min, src, &sp);
		old_sp_re = sp;		/* Make this search's parameter the defaults for the next. */
	}
	return 0;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-12 -	Extend the selection to include all rows having styles that are siblings to
**		the currently selected row(s). Typically useful to select e.g. all images
**		after selecting one, and things like that. Assumes the style tree layout is
**		sensible (i.e. rather deep and with "abstract" parents).
*/
gint cmd_selectext(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	GSList	*slist, *iter;
	guint	i;

	slist = dp_get_selection(src);

	for(iter = slist; iter != NULL; iter = g_slist_next(iter))
	{
		for(i = 0; i < src->dir.num_rows; i++)
		{
			if(DP_SEL_INDEX(src, iter) == i)
				continue;

			if(stl_styleinfo_style_siblings(min->cfg.style, DP_SEL_TYPE(iter)->style, DP_ROW_TYPE(&src->dir.row[i])->style))
				dp_select(src, i);
		}
	}
	dp_free_selection(slist);

	return 1;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-06-12 -	Perform a suffix selection. If there is a mouse button event available, use
**		it to determine which row has the desired suffix. If not, use the focus if
**		one exists. Else, do nothing.
*/
gint cmd_selectsuffix(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	GdkEventButton	*evt;
	gint		row = -1;
	const gchar	*suff = NULL;

	if((suff = car_bareword_get(ca, 0)) != NULL)
		;
	if(suff == NULL)
	{
		if((evt = (GdkEventButton *) evt_event_get(GDK_BUTTON_PRESS)) != NULL)
		{
			gint	col;

			if(!gtk_clist_get_selection_info(GTK_CLIST(src->list), evt->x, evt->y, &row, &col) || (row > src->dir.num_rows))
				row = -1;
		}
		else
			row = src->focus_row;
		if(row != -1)
			suff = stu_strcasechr(DP_ROW_NAME(&src->dir.row[row]), '.');
	}

	/* Did we find a row? If we did, extract its suffix, and select all matching rows. */
	if(suff != NULL)
	{
		guint		action, i;
		void		(*afunc)(DirPane *dp, gint row);

		action = car_keyword_get_enum(ca, "action", 0, "select", "unselect", "toggle", NULL);
		afunc = (action == 0) ? dp_select : (action == 1) ? dp_unselect : dp_toggle;

		for(i = 0; i < src->dir.num_rows; i++)
		{
			if(stu_has_suffix(DP_ROW_NAME(&src->dir.row[i]), suff))
				afunc(src, i);
		}
	}
	return 1;
}

/* 1999-06-12 -	Alter the selected state of all rows having the same type as a controlling
**		row. The latter is either the one most recently clicked, or the focused row.
** NOTE NOTE	This command replaces a command that had the same name, but that I think
**		wasn't very useful. If you disagree, get in touch. Thanks.
*/
gint cmd_selecttype(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	GdkEventButton	*evt;
	const gchar	*tname;
	gint		row = -1;
	GList		*types = NULL;

	if((tname = car_bareword_get(ca, 0U)) != NULL)
		types = typ_type_lookup_glob(min->cfg.type, tname);
	else
	{
		if((evt = (GdkEventButton *) evt_event_get(GDK_BUTTON_PRESS)) != NULL)
		{
			gint	col;

			if(!gtk_clist_get_selection_info(GTK_CLIST(src->list), evt->x, evt->y, &row, &col) || (row > src->dir.num_rows))
				row = -1;
		}
		else
			row = src->focus_row;
		if(row != -1)
			types = g_list_append(NULL, (gpointer) DP_ROW_TYPE(&src->dir.row[row]));
	}

	if(types != NULL)
	{

		guint	action, i;
		void	(*afunc)(DirPane *dp, gint row);

		action = car_keyword_get_enum(ca, "action", 0, "select", "unselect", "toggle", NULL);
		afunc  = (action == 0) ? dp_select : (action == 1) ? dp_unselect : dp_toggle;

		for(i = 0; i < src->dir.num_rows; i++)
		{
			if(g_list_find(types, (gpointer) DP_ROW_TYPE(&src->dir.row[i])) != NULL)
				afunc(src, i);
		}
		g_list_free(types);
	}
	return 1;
}

/* ----------------------------------------------------------------------------------------- */

#if 0
struct cmp_rules {
	gboolean	nocase;

	gint		(*scmp)(const gchar *s1, const gchar *s2);
};

/* 2001-09-23 -	Compare <row> against all rows in <dst>, according to rules specified in <dr>.
** 		Returns TRUE if there is a match, i.e. if a file named <row> exists in <dst>
**		already.
*/
static gboolean cmp_exec(const DirRow *row, const DirPane *dst, const struct cmp_rules *dr)
{
	const gchar	*name;
	gboolean	ret;
	gint		i;
	guint32		match;

	name = DP_ROW_NAME(row);
	for(i = 0; i < dst->dir.num_rows; i++)
	{
		if(dr->scmp(name, DP_ROW_NAME(&dst->dir.row[i])) != 0)
			continue;
		/* At this point, we have a name match. Furher requirements? */
		return TRUE;
	}
	return FALSE;
}

gint cmd_selectcmp(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	static struct cmp_rules	rules = { FALSE };
	gint			i;

	/* Select string comparison function to use. */
	rules.scmp  = rules.nocase ? g_strcasecmp : strcmp;

	for(i = 0; i < src->dir.num_rows; i++)
	{
		if(cmp_exec(&src->dir.row[i], dst, &rules))
			dp_select(src, i);
	}
	return 1;
}
#endif
/* ----------------------------------------------------------------------------------------- */

typedef enum { SHELL_BOOLEAN } ShellMode;

typedef struct {
	ShellMode	mode;
} ShellRule;

static int shell_select(char **argv, const char *dir, const ShellRule *rule)
{
	pid_t	child;

	if(argv == NULL || argv[0] == NULL || rule == NULL)
		return 0;

	if((child = fork()) >= 0)
	{
		if(child == 0)	/* In child? */
		{
			fut_cd(dir, NULL, 0u);
			execvp(argv[0], argv);
			usleep(50000);
			_exit(-1);
		}
		else if(child > 0)		/* In parent. */
		{
			int	stat = 0;

			waitpid(child, &stat, 0);
			if(WIFEXITED(stat))
			{
				stat = WEXITSTATUS(stat);
				if(rule->mode == SHELL_BOOLEAN)
					return stat == 0;
				else
					fprintf(stderr, __FILE__ ": Not handling shell select mode %d\n", rule->mode);
			}
			else
				fprintf(stderr, __FILE__ ": Child command %s crashed, not interpreting return status\n", argv[0]);
		}
	}
	return 0;
}

/* 2006-04-24 -	Select based on return value of shell command. */
gint cmd_selectshell(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	SelParamShell	sp;
	const gchar	*str;
	GString		*cmd;
	guint		i, column;
	gchar		**argv;
	gint		reply = DLG_POSITIVE;

	/* Start from previous set of parameters. */
	sp = old_sp_shell;
	selparam_set_from_cmdarg(&sp.basic, ca);
	/* If bareword found, use it as command. Else, pop up the big old dialog. */
	if((str = car_bareword_get(ca, 0)) != NULL)
	{
		if(sp.cmd == NULL)
			sp.cmd = g_string_new(str);
		else
			g_string_assign(sp.cmd, str);
	}
	else
	{
		GtkWidget	*vbox, *opt_menu[4], *label, *entry;
		guint		*mptr[sizeof opt_menu / sizeof *opt_menu];
		Dialog		*dlg;
	
		selparam_get_value_pointers(mptr, &sp.basic);
		vbox = gtk_vbox_new(FALSE, 0);
		selparam_init_menus(src, vbox, opt_menu, mptr);
		label = gtk_label_new(_("Enter shell command to run. The command\nwill have the selected content appended.\nAction is performed on successful exit."));
		gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
		entry = gtk_entry_new_with_max_length(1024);
		if(sp.cmd != NULL)
			gtk_entry_set_text(GTK_ENTRY(entry), sp.cmd->str);
		gtk_box_pack_start(GTK_BOX(vbox), entry, FALSE, FALSE, 0);
		dlg = dlg_dialog_sync_new(vbox, _("Select using shell command"), _("OK|Cancel"));
		gtk_entry_select_region(GTK_ENTRY(entry), 0, 1 << 30);
		gtk_widget_grab_focus(entry);
		reply = dlg_dialog_sync_wait(dlg);
		if(reply == DLG_POSITIVE)
		{
			selparam_read_menus(mptr, opt_menu);
			if(sp.cmd == NULL)
				sp.cmd = g_string_new(gtk_entry_get_text(GTK_ENTRY(entry)));
			else
				g_string_assign(sp.cmd, gtk_entry_get_text(GTK_ENTRY(entry)));
		}
		dlg_dialog_sync_destroy(dlg);
	}

	if(reply == DLG_POSITIVE)
	{
		SelAction	sel_action;
		gchar		*text;

		column = compute_column(src, sp.basic.content);
		cmd = g_string_sized_new(4096);		/* Static for simplicity. */
		sel_action = select_action_func(sp.basic.action);
		for(i = 0; i < src->dir.num_rows; i++)
		{
			if(!filter_set(src, i, &sp.basic) || !filter_type(src, i, &sp.basic))
				continue;
			g_string_assign(cmd, sp.cmd->str);
			g_string_append_c(cmd, ' ');
			text = NULL;
			gtk_clist_get_text(GTK_CLIST(src->list), i, column, &text);
			if(text == NULL)
				continue;
			g_string_append(cmd, text);
			if(cmd->str[0] == '\0' || cmd->str[0] == ' ')
				continue;
			argv = cpr_parse(min, cmd->str);
			if(argv != NULL && argv[0] != NULL)
			{
				const ShellRule	rule = { SHELL_BOOLEAN };

				if(shell_select(argv, src->dir.path, &rule))
					sel_action(src, i);
				g_free(argv);
			}
		}
		g_string_free(cmd, TRUE);
		old_sp_shell = sp;		/* Remember parameters for next time. */
	}
	return 0;
}

/* ----------------------------------------------------------------------------------------- */

/* 1998-09-20 -	The UnselectFirst command. In wrong module?
** 1998-12-16 -	This was freeze/thaw bracketed, for no good reason. Didn't work.
*/
gint cmd_unselectfirst(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	GSList	*slist;

	if((slist = dp_get_selection(src)) != NULL)
	{
		dp_unselect(src, DP_SEL_INDEX(src, slist));
		dp_free_selection(slist);
	}
	return 1;
}
