/*
** 1999-09-11 -	A fun Rename command, with support for both simplistic replacement in filenames
**		as well as powerful regular expression replacement thingie. The former is handy
**		for changing the extension in all selected names from ".jpg" to ".jpeg", for
**		example. The latter RE mode can be used to do rather complex mappings and trans-
**		forms on the filenames. For example, consider having a set of 100 image files,
**		named "frame0.gif", "frame1.gif", ..., "frame99.gif". You want to rename these
**		to include a movie index (11), and also (of course) change the extension to ".png".
**		No problem. Just enter "^frame([0-9]+)\.gif$" in the 'From' text entry field
**		on the Reg Exp page, and "frame-11-$1.png" in the 'To' entry, and bam! The point
**		here is that you can use up to 9 parenthesized (?) subexpressions, and then access
**		the text that matched each with $n (n is a digit, 1 to 9) in the 'To' string.
**		If you've programmed any Perl, you might be familiar with the concept.
*/

#include "gentoo.h"

#include <ctype.h>
#include <regex.h>

#include "cmd_delete.h"
#include "dialog.h"
#include "dirpane.h"
#include "errors.h"
#include "fileutil.h"
#include "overwrite.h"
#include "strutil.h"

#include "cmd_renamere.h"

#define	CMD_ID	"renamere"

/* ----------------------------------------------------------------------------------------- */

typedef struct {
	Dialog		*dlg;

	GtkWidget	*nbook;

	/* Simple replace mode. */
	GtkWidget	*r_vbox;
	GtkWidget	*r_from;
	GtkWidget	*r_to;
	GtkWidget	*r_cnocase;
	GtkWidget	*r_cglobal;

	/* Full RE matching mode. */
	GtkWidget	*re_vbox;
	GtkWidget	*re_from;
	GtkWidget	*re_to;
	GtkWidget	*re_cnocase;

	/* Mapping mode (for character substitutions). */
	GtkWidget	*map_vbox;
	GtkWidget	*map_from;
	GtkWidget	*map_to;
	GtkWidget	*map_remove;

	/* Case-conversion mode. */
	GtkWidget	*case_vbox;
	GtkWidget	*case_lower;

	MainInfo	*min;
	DirPane		*src;
	gint		page;
	GSList		*selection;
} RenREInfo;

/* ----------------------------------------------------------------------------------------- */

static gint do_rename(MainInfo *min, DirPane *dp, GSList **iter, GString *nn, gint *err)
{
	if(strcmp(DP_SEL_NAME(*iter), nn->str) != 0)
	{
		OvwRes	ores;

		ores = ovw_overwrite_file(min, nn->str, NULL);
		if(ores == OVW_SKIP)
			return 0;
		else if(ores == OVW_CANCEL)
		{
			*iter = NULL;
			return -1;
		}
		else if(ores == OVW_PROCEED_FILE)
		{
			if(!del_delete_file(min, nn->str))
			{
				*iter = NULL;
				return -1;
			}
		}
		else if(ores == OVW_PROCEED_DIR)
		{
			if(!del_delete_dir(min, nn->str, FALSE))
			{
				*iter = NULL;
				return -1;
			}
		}
		if(rename(DP_SEL_NAME(*iter), nn->str) != 0)
		{
			*err = errno;
			dp_rescan_post_cmd(dp);
			err_set(min, *err, CMD_ID, DP_SEL_NAME(*iter));
		}
		else
			dp_unselect(dp, DP_SEL_INDEX(dp, *iter));
	}
	return errno;
}

/* ----------------------------------------------------------------------------------------- */

static gint rename_simple(RenREInfo *ri)
{
	const gchar	*from, *to, *ptr, *base;
	gint		err = 0;
	gsize		flen;
	GString		*nn;
	GSList		*iter;
	gboolean	nocase, global;

	if((from = gtk_entry_get_text(GTK_ENTRY(ri->r_from))) == NULL)
		return -1;
	flen = strlen(from);

	to = gtk_entry_get_text(GTK_ENTRY(ri->r_to));

	nocase = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ri->r_cnocase));
	global = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ri->r_cglobal));

	ovw_overwrite_begin(ri->min, _("\"%s\" Already Exists - Proceed With Rename?"), 0U);
	nn = g_string_new("");
	for(iter = ri->selection; !err && iter != NULL; iter = g_slist_next(iter))
	{
		g_string_truncate(nn, 0);

		base = DP_SEL_NAME(iter);
		while((ptr = nocase ? stu_strcasechr(base, *from) : strchr(base, *from)) != NULL)
		{
			for(; base < ptr; base++)
				g_string_append_c(nn, *base);
			if((nocase ? g_strncasecmp(ptr, from, flen) : strncmp(ptr, from, flen)) == 0)
			{
				g_string_append(nn, to);
				base += flen;
				if(!global)
					break;
			}
			else
				g_string_append_c(nn, *base++);
		}
		g_string_append(nn, base);
		if(strchr(nn->str, G_DIR_SEPARATOR) != NULL)
			continue;
		do_rename(ri->min, ri->src, &iter, nn, &err);
	}
	g_string_free(nn, TRUE);
	ovw_overwrite_end(ri->min);

	return err;
}

/* ----------------------------------------------------------------------------------------- */

/* 1999-09-11 -	Go through <def>, copying characters to <str>. If the sequence $n is found,
**		where n is a digit 1..9, replace that with the n:th piece of <src>, as described
**		by <match>. To get a single dollar in the output, write $$ in <def>.
**		Returns TRUE if the entire <def> was successfully interpolated, FALSE on error.
*/
static gboolean interpolate(GString *str, const gchar *def, const gchar *src, const regmatch_t *match)
{
	const gchar	*ptr;

	g_string_truncate(str, 0);
	for(ptr = def; *ptr;)
	{
		if(*ptr == '$')
		{
			ptr++;
			if(isdigit((guchar) *ptr))
			{
				gint	slot, i;

				slot = *ptr++ - '0';
				if((i = match[slot].rm_so) != -1)
				{
					for(; i < match[slot].rm_eo; i++)
						g_string_append_c(str, src[i]);
				}
				else
					return FALSE;
			}
			else if(*ptr == '$')
				g_string_append_c(str, '$'), ptr++;
			else
				return FALSE;
		}
		else
			g_string_append_c(str, *ptr++);
	}
	return TRUE;
}

static gint rename_regexp(RenREInfo *ri)
{
	const gchar	*fromdef, *to;
	gint		reerr, reflags, err = 0;
	regex_t		fromre;

	if((fromdef = gtk_entry_get_text(GTK_ENTRY(ri->re_from))) == NULL)
		return -1;
	if((to = gtk_entry_get_text(GTK_ENTRY(ri->re_to))) == NULL)
		return -1;

	reflags = REG_EXTENDED | gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ri->re_cnocase));
	if((reerr = regcomp(&fromre, fromdef, reflags)) == 0)
	{
		GSList		*iter;
		GString		*nn;
		regmatch_t	match[10];

		ovw_overwrite_begin(ri->min, _("\"%s\" Already Exists - Proceed With Rename?"), 0U);
		nn = g_string_new("");
		for(iter = ri->selection; iter != NULL; iter = g_slist_next(iter))
		{
			if(regexec(&fromre, DP_SEL_NAME(iter), sizeof match / sizeof match[0], match, 0) == 0)
			{
				if(interpolate(nn, to, DP_SEL_NAME(iter), match) && strchr(nn->str, G_DIR_SEPARATOR) == NULL)
					do_rename(ri->min, ri->src, &iter, nn, &err);
			}
		}
		g_string_free(nn, TRUE);
		ovw_overwrite_end(ri->min);
	}
	else
	{
		gchar	buf[128];

		strcpy(buf, _("Regular expression error:\n"));
		regerror(reerr, &fromre, buf + strlen(buf), sizeof buf - strlen(buf) - 1);
		dlg_dialog_async_new_error(buf);
	}
	return err;
}

/* ----------------------------------------------------------------------------------------- */

static int rename_map(RenREInfo *ri)
{
	const char	*fromdef, *todef, *remdef, *p;
	GSList		*iter;
	GString		*nn;
	gint		err = 0;
	gsize		fl, tl, rl, i, j;

	if((fromdef = gtk_entry_get_text(GTK_ENTRY(ri->map_from))) == NULL)
		return -1;
	if((todef = gtk_entry_get_text(GTK_ENTRY(ri->map_to))) == NULL)
		return -1;
	if(strchr(todef, G_DIR_SEPARATOR) != NULL)	/* As always, prevent rogue moves. */
		return -1;
	if((remdef = gtk_entry_get_text(GTK_ENTRY(ri->map_remove))) == NULL)
		return -1;

	fl = strlen(fromdef);
	tl = strlen(todef);
	rl = strlen(remdef);

	ovw_overwrite_begin(ri->min, _("\"%s\" Already Exists - Proceed With Rename?"), 0U);
	nn = g_string_new("");
	for(iter = ri->selection; iter != NULL; iter = g_slist_next(iter))
	{
		/* Step 1: set the dynamic string to the input filename. */
		g_string_assign(nn, DP_SEL_NAME(iter));
		/* Step 2: perform character mapping, as needed. */
		for(i = 0; i < nn->len; i++)
		{
			if((p = strchr(fromdef, nn->str[i])) != NULL)
			{
				j = p - fromdef;
				if(j < tl)
					nn->str[i] = todef[j];
			}
		}
		/* Step 3: remove any characters found in the 'remove' string. */
		for(i = 0; i < nn->len;)
		{
			if(strchr(remdef, nn->str[i]) != NULL)
				g_string_erase(nn, i, 1);
			else
				i++;
		}
		do_rename(ri->min, ri->src, &iter, nn, &err);
	}
	g_string_free(nn, TRUE);
	ovw_overwrite_end(ri->min);

	return err;
}

/* ----------------------------------------------------------------------------------------- */

static int rename_case(RenREInfo *ri)
{
	gboolean	lower = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ri->case_lower));
	GString		*nn;
	GSList		*iter;
	gint		err = 0;

	ovw_overwrite_begin(ri->min, _("\"%s\" Already Exists - Proceed With Rename?"), 0U);
	nn = g_string_new("");
	for(iter = ri->selection; iter != NULL; iter = g_slist_next(iter))
	{
		/* Step 1: set the dynamic string to the input filename. */
		g_string_assign(nn, DP_SEL_NAME(iter));
		/* Step 2: perform case conversion. */
		if(lower)
			g_string_down(nn);
		else
			g_string_up(nn);
		/* Step 3: if resulting name is different, rename on disk. */
		do_rename(ri->min, ri->src, &iter, nn, &err);
	}
	g_string_free(nn, TRUE);
	ovw_overwrite_end(ri->min);

	return err;
}

/* ----------------------------------------------------------------------------------------- */

static void evt_nbook_switchpage(GtkWidget *wid, GtkNotebookPage *page, gint page_num, gpointer user)
{
	RenREInfo	*ri = user;

	ri->page = page_num;
	gtk_widget_grab_focus(ri->page ? ri->re_from : ri->r_from);
}

gint cmd_renamere(MainInfo *min, DirPane *src, DirPane *dst, const CmdArg *ca)
{
	GtkWidget		*label, *table, *wid;
	static RenREInfo	ri = { NULL };

	ri.min	= min;
	ri.src	= src;

	if((ri.selection = dp_get_selection(ri.src)) == NULL)
		return 1;

	if(ri.dlg == NULL)
	{
		ri.nbook = gtk_notebook_new();
		gtk_signal_connect(GTK_OBJECT(ri.nbook), "switch-page", GTK_SIGNAL_FUNC(evt_nbook_switchpage), &ri);

		/* Build the 'Simple' mode's GUI. */
		ri.r_vbox = gtk_vbox_new(FALSE, 0);
		label = gtk_label_new(_("Look for substring in all filenames, and replace\nit with another string."));
		gtk_box_pack_start(GTK_BOX(ri.r_vbox), label, FALSE, FALSE, 0);
		table = gtk_table_new(2, 2, FALSE);
		label = gtk_label_new(_("Replace"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,  0,0,0,0);
		ri.r_from = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.r_from, 1, 2, 0, 1, GTK_EXPAND|GTK_FILL,0,0,0);
		label = gtk_label_new(_("With"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,  0,0,0,0);
		ri.r_to = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.r_to, 1, 2, 1, 2, GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_box_pack_start(GTK_BOX(ri.r_vbox), table, FALSE, FALSE, 0);
		table = gtk_table_new(1, 2, FALSE);
		ri.r_cnocase = gtk_check_button_new_with_label(_("Ignore Case?"));
		gtk_table_attach(GTK_TABLE(table), ri.r_cnocase, 0, 1, 0, 1, GTK_EXPAND|GTK_FILL,0,0,0);
		ri.r_cglobal = gtk_check_button_new_with_label(_("Replace All?"));
		gtk_table_attach(GTK_TABLE(table), ri.r_cglobal, 1, 2, 0, 1, GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_box_pack_start(GTK_BOX(ri.r_vbox), table, FALSE, FALSE, 0);

		gtk_notebook_append_page(GTK_NOTEBOOK(ri.nbook), ri.r_vbox, gtk_label_new(_("Simple")));

		/* Build the 'Reg Exp' mode's GUI. */
		ri.re_vbox = gtk_vbox_new(FALSE, 0);
		label = gtk_label_new(	_("Execute the 'From' RE on each filename, storing\n"
					"parenthesised subexpression matches. Then replace\n"
					"any occurance of $n in 'To', where n is the index\n"
					"(counting from 1) of a subexpression, with the text\n"
					"that matched, and use the result as a new filename."));

		gtk_box_pack_start(GTK_BOX(ri.re_vbox), label, FALSE, FALSE, 0);
		table = gtk_table_new(2, 2, FALSE);
		label = gtk_label_new(_("From"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,  0,0,0,0);
		ri.re_from = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.re_from, 1, 2, 0, 1,  GTK_EXPAND|GTK_FILL,0,0,0);
		label = gtk_label_new(_("To"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,  0,0,0,0);
		ri.re_to = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.re_to, 1, 2, 1, 2,  GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_box_pack_start(GTK_BOX(ri.re_vbox), table, FALSE, FALSE, 0);
		ri.re_cnocase = gtk_check_button_new_with_label(_("Ignore Case?"));
		gtk_box_pack_start(GTK_BOX(ri.re_vbox), ri.re_cnocase, FALSE, FALSE, 0);

		gtk_notebook_append_page(GTK_NOTEBOOK(ri.nbook), ri.re_vbox, gtk_label_new(_("Reg Exp")));
		gtk_widget_show_all(ri.re_vbox);

		/* Build the 'Map' mode's GUI. */
		ri.map_vbox = gtk_vbox_new(FALSE, 0);
		label = gtk_label_new(_("Look for each character in the 'From' string, and replace\n"
					"any hits with the corresponding character in the 'To' string.\n"
					"Then, any characters in the 'Remove' string are removed from\n"
					"the filename, and the result used as the new name for each file."));
		gtk_box_pack_start(GTK_BOX(ri.map_vbox), label, FALSE, FALSE, 0);
		table = gtk_table_new(3, 2, FALSE);
		label = gtk_label_new(_("From"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,  0,0,0,0);
		ri.map_from = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.map_from, 1, 2, 0, 1,  GTK_EXPAND|GTK_FILL,0,0,0);
		label = gtk_label_new(_("To"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,  0,0,0,0);
		ri.map_to = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.map_to, 1, 2, 1, 2,  GTK_EXPAND|GTK_FILL,0,0,0);
		label = gtk_label_new(_("Remove"));
		gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,  0,0,0,0);
		ri.map_remove = gtk_entry_new();
		gtk_table_attach(GTK_TABLE(table), ri.map_remove, 1, 2, 2, 3,  GTK_EXPAND|GTK_FILL,0,0,0);
		gtk_box_pack_start(GTK_BOX(ri.map_vbox), table, FALSE, FALSE, 0);
		gtk_notebook_append_page(GTK_NOTEBOOK(ri.nbook), ri.map_vbox, gtk_label_new(_("Map")));

		/* Build the 'Case' mode's GUI. */
		ri.case_vbox = gtk_vbox_new(FALSE, 0);
		label = gtk_label_new(_("Convert all characters in filename to either\nall-lower or all-upper case."));
		gtk_box_pack_start(GTK_BOX(ri.case_vbox), label, FALSE, FALSE, 0);
		ri.case_lower = gtk_radio_button_new_with_label(NULL, _("Lower Case?"));
		gtk_box_pack_start(GTK_BOX(ri.case_vbox), ri.case_lower, FALSE, FALSE, 0);
		wid = gtk_radio_button_new_with_label(gtk_radio_button_group(GTK_RADIO_BUTTON(ri.case_lower)), _("Upper Case?"));
		gtk_box_pack_start(GTK_BOX(ri.case_vbox), wid, FALSE, FALSE, 0);
		gtk_widget_show_all(ri.map_vbox);
		gtk_notebook_append_page(GTK_NOTEBOOK(ri.nbook), ri.case_vbox, gtk_label_new(_("Case")));

		ri.dlg = dlg_dialog_sync_new(ri.nbook, _("RenameRE"), NULL);

		ri.page = 0;
	}
	gtk_notebook_set_page(GTK_NOTEBOOK(ri.nbook), ri.page);
	gtk_widget_grab_focus(ri.page ? ri.re_from : ri.r_from);

	if(dlg_dialog_sync_wait(ri.dlg) == DLG_POSITIVE)
	{
		gchar	old[PATH_MAX];
		gint	err = 0;

		err_clear(ri.min);
		if(fut_cd(src->dir.path, old, sizeof old))
		{
			if(ri.page == 0)
				err = rename_simple(&ri);
			else if(ri.page == 1)
				err = rename_regexp(&ri);
			else if(ri.page == 2)
				err = rename_map(&ri);
			else
				err = rename_case(&ri);
			fut_cd(old, NULL, 0U);
		}
		else
			err_set(ri.min, errno, "CD", src->dir.path);
		if(!err && !errno)
			dp_rescan_post_cmd(ri.src);
		else
			err_show(ri.min);
	}
	dp_free_selection(ri.selection);
	ri.selection = NULL;

	return 1;
}
